/*******************************************************************
 *  Copyright (c) 2007-2014 Jetico, Inc., Finland
 *  All rights reserved.
 *
 *  File:          schemes.c
 *
 *  Description:   wiping schemes
 *
 *  Author:        Alexander Pichuev
 *
 *  Created:       10-Sep-2007
 *
 *  Revision:      $Id: schemes.c 381 2015-04-24 06:15:13Z nail $ 
 *
 *
 *******************************************************************/
#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include "wipe.h"
#include "options.h"
#include "log.h"
#include "schemes.h"

static pass_s pg_passT[] = {
    { 0, 0, 0, PASS_RANDOM    },  /* 1  random */
    { 0, 0, 0, PASS_RANDOM    },  /* 2  random */
    { 0, 0, 0, PASS_RANDOM    },  /* 3  random */
    { 0, 0, 0, PASS_RANDOM    },  /* 4  random */
    { 1, 0, "\x55",           },  /* 5  RLL MFM */
    { 1, 0, "\xaa",           },  /* 6  RLL MFM */
    { 3, 0, "\x92\x49\x24",   },  /* 7  RLL MFM */
    { 3, 0, "\x49\x24\x92",   },  /* 8  RLL MFM */
    { 3, 0, "\x24\x92\x49",   },  /* 9  RLL MFM */
    { 1, 0, "\x00",           },  /* 10 RLL */
    { 1, 0, "\x11",           },  /* 11 RLL */
    { 1, 0, "\x22",           },  /* 12 RLL */
    { 1, 0, "\x33",           },  /* 13 RLL */
    { 1, 0, "\x44",           },  /* 14 RLL */
    { 1, 0, "\x55",           },  /* 15 RLL */
    { 1, 0, "\x66",           },  /* 16 RLL */
    { 1, 0, "\x77",           },  /* 17 RLL */
    { 1, 0, "\x88",           },  /* 18 RLL */
    { 1, 0, "\x99",           },  /* 19 RLL */
    { 1, 0, "\xaa",           },  /* 20 RLL */
    { 1, 0, "\xbb",           },  /* 21 RLL */
    { 1, 0, "\xcc",           },  /* 22 RLL */
    { 1, 0, "\xdd",           },  /* 23 RLL */
    { 1, 0, "\xee",           },  /* 24 RLL */
    { 1, 0, "\xff",           },  /* 25 RLL */
    { 3, 0, "\x92\x49\x24"    },  /* 26 RLL MFM */
    { 3, 0, "\x49\x24\x92"    },  /* 27 RLL MFM */
    { 3, 0, "\x24\x92\x49"    },  /* 28 RLL MFM */
    { 3, 0, "\x6d\xb6\xdb"    },  /* 29 RLL */
    { 3, 0, "\xb6\xdb\x6d"    },  /* 30 RLL */
    { 3, 0, "\xdb\x6d\xb6"    },  /* 31 RLL */
    { 0, 0, 0, PASS_RANDOM    },  /* 32 random */
    { 0, 0, 0, PASS_RANDOM    },  /* 33 random */
    { 0, 0, 0, PASS_RANDOM    },  /* 34 random */
    { 0, 1, 0, PASS_RANDOM    }   /* 35 random with verification */
};

static pass_s dod_passT[] = {
    { 1, 0, "\x35",         },
    { 1, 0, "\xca",         },
    { 1, 0, "\x97",         },
    { 1, 0, "\x68",         },
    { 1, 0, "\xac",         },
    { 1, 0, "\x53",         },
    { 0, 1, 0, PASS_RANDOM  }
};

static pass_s bci_passT[] = {
    { 0, 0, 0, PASS_ZERO   },
    { 1, 0, "\xFF", 0      },
    { 0, 0, 0, PASS_ZERO   },
    { 1, 0, "\xFF", 0      },
    { 0, 0, 0, PASS_ZERO   },
    { 1, 0, "\xFF", 0      },
    { 1, 1, "\xAA", 0      }
};

static pass_s test_passT[] = {
    { 0, 1, 0, PASS_TEST }
};

static pass_s zero_passT[] = {
    { 0, 1, 0, PASS_ZERO }
};

static pass_s doe_passT[] = {
    { 0, 0, 0, PASS_RANDOM },
    { 0, 0, 0, PASS_RANDOM },
    { 0, 1, 0, PASS_ZERO   }
};

static pass_s schneier_passT[] = {
    { 1, 0, "\xFF", 0      },
    { 0, 0, 0, PASS_ZERO   },
    { 0, 0, 0, PASS_RANDOM },
    { 0, 0, 0, PASS_RANDOM },
    { 0, 0, 0, PASS_RANDOM },
    { 0, 0, 0, PASS_RANDOM },
    { 0, 1, 0, PASS_RANDOM }
};

int init_scheme(wipe_scheme_t *scheme)
{
	int i;

	if (!scheme) {
		return -1;
	}

	switch (o_scheme) {
	case SCHEME_FILE:
		i = load_scheme(o_scheme_file, scheme);
		if (o_scheme_file) {
			free(o_scheme_file);
		}
		o_pas_num = scheme->num_passes;
		return i;

	case SCHEME_GUTMANN:
		scheme->name = "Peter Gutmann's";
		scheme->num_passes = 35;
		scheme->pass = pg_passT;
		scheme->builtin = TRUE;
		break;

	case SCHEME_DOE:
		scheme->name = "US DoE";
		scheme->num_passes = 3;
		scheme->pass = doe_passT;
		scheme->builtin = TRUE;
		break;

	case SCHEME_BCI:
		scheme->name = "German BCI/VSITR";
		scheme->num_passes = 7;
		scheme->pass = bci_passT;
		scheme->builtin = TRUE;
		break;

	case SCHEME_DOD:
		scheme->name = "US DoD 5220-22M";
		scheme->num_passes = o_pas_num;
		if (7 == scheme->num_passes) {
			scheme->pass = dod_passT;
			scheme->builtin = TRUE;
		} else {
			scheme->pass = (pass_s *)malloc(sizeof(pass_s)*scheme->num_passes);
			if (NULL == scheme->pass) {
				log_message(LOG_FATAL, "Out of memory when trying to build a scheme\n");
				return -1;
			}
			for (i = 0; i < scheme->num_passes - 1; i++) {
				scheme->pass[i] = dod_passT[i%6];
			}
			scheme->pass[scheme->num_passes - 1] = dod_passT[6];
			scheme->name = strdup("Modified US DoD 5220-22M");
			scheme->builtin = FALSE;
		}
		break;

	case SCHEME_SCHNEIER:
		scheme->name = "Bruce Schneier's";
		scheme->num_passes = 7;
		scheme->pass = schneier_passT;
		scheme->builtin = TRUE;
		break;

	case SCHEME_TEST:
		scheme->name = "Test";
		scheme->num_passes = 1;
		scheme->pass = test_passT;
		scheme->builtin = TRUE;
		break;

	case SCHEME_ZERO:
		scheme->name = "Zero";
		scheme->num_passes = 1;
		scheme->pass = zero_passT;
		scheme->builtin = TRUE;
		break;


	default:
		log_message(LOG_FATAL, "Unknown wiping scheme %d requested\n", o_scheme);
		return -1;
	}

	o_pas_num = scheme->num_passes;

	return 0;
}
/*
void cleanup_scheme(wipe_scheme_t *scheme) 
{
	if (!scheme) {
		return;
	}

	if (!scheme->builtin && scheme->pass) {
		free(scheme->pass);
		scheme->pass = NULL;
	}
}
*/
void cleanup_scheme(wipe_scheme_t *s)
{
        unsigned int i;

        if (!s->builtin) {
                free(s->name);
                for (i = 0; i < s->num_passes; i++) {
                    if (s->pass[i].pat) {
                        free(s->pass[i].pat);
                    }
                }
                free(s->pass);
                memset(s, 0, sizeof(wipe_scheme_t));
        }
}


int print_pass_name(char *buf, int buf_len, wipe_scheme_t *scheme, int pass)
{
	int i, x, len;
	
	if (!buf || 2 > buf_len) {
		return -1;
	} else if (!scheme || pass >= scheme->num_passes) {
		buf[0] = 0;
		return 1;
	}
	
	len = scheme->pass[pass].len;
	if (len) {
		for (i = 0, x = 0; i < len; i++) { 
			x += snprintf(buf+x, buf_len-x, "%02X%s", 0xFF & scheme->pass[pass].pat[i], i==len-1 ? "":", ");
		}
		return x;
	}
	
	switch (scheme->pass[pass].type) {
	case PASS_RANDOM:
		return snprintf(buf, buf_len, "random%s", o_use_buff ? " pattern" : "" );
	case PASS_COMPLEMENT:
		return snprintf(buf, buf_len, "complementary");
	case PASS_ZERO:
		return snprintf(buf, buf_len, "zeroes");
	case PASS_TEST:
		return snprintf(buf, buf_len, "test");
	default: ;
	}
	buf[0] = 0;
	return 1;
}


static int wipe_read_pattern(char *buf, int buf_len, pass_s *pi) 
{
	char	*str, *tok, *tmp = NULL;
	int	x, sz, cnt, val;
	

	str = strchr(buf, '.');
	if (!str) {
		return 0;
	}
	
	sz = 0;
	if (pi) {
		tmp = (char *)malloc(MAX_PATTERN_LEN);
		if (!tmp) {
			return 0;
		}
	}
	
	tok = strtok(str+1, ",");
	while (tok) {
		str = strchr(tok, '+');
		if (str) {
			x = sscanf(str, "+%u %x", &cnt, &val);
		} else {
			x = sscanf(tok, "%x", &val);
			if (1 == x) {
				cnt = 1;
				x = 2;
			}
		}
		if (2 == x) {
			if (sz+cnt >= MAX_PATTERN_LEN) {
				cnt = MAX_PATTERN_LEN - sz;
			}
			if (tmp) {
				memset(tmp+sz, val & 0xFF, cnt);
			}
			sz += cnt;
		}
		if (sz >= MAX_PATTERN_LEN) {
			break;
		}
		tok = strtok(NULL, ",");
	}
	
	if (pi) {
		pi->len = sz;
		pi->pat = (char *)malloc(sz);
		if (!pi->pat) {
			pi->pat = tmp;
		} else {
			memcpy(pi->pat, tmp, sz);
			free(tmp);
		}
	}
	
	return sz > 0 ? 1 : 0;

}

static int wipe_read_pass(char *buf, int buf_len, pass_s *pi) 
{
	int  i, ret, test = 0;
	pass_s	p;
	
	if (0 == pi) {
		pi = &p;
		test = 1;
	}

	// Make buffer lowercase
	for (i = 0; i < buf_len; i++) {
		if (!buf[i])
			break;
		buf[i] = tolower(buf[i]);
	}		
	
	// Look for keywords
	pi->len = 0;
	ret = 0;
	if (strstr(buf, "random")) {
		pi->type = PASS_RANDOM;
		ret = 1;
	} else if (strstr(buf, "complement")) {
		pi->type = PASS_COMPLEMENT;
		ret = 1;
	} else if (strstr(buf, "test")) {
		pi->type = PASS_TEST;
		ret = 1;
	} else if (strstr(buf, "zero")) {
		pi->type = PASS_ZERO;
		ret = 1;
	} else {
		ret = wipe_read_pattern(buf, buf_len, test ? NULL : pi);
		pi->type = PASS_PATTERN;
	}

	if (ret) {
		pi->verify = strstr(buf, "verify") ? 1 : 0;
	}
	
	return ret;
}

int load_scheme(char* path, wipe_scheme_t* out_info)
{
	FILE	*f;
	char	buf[1024], *p1, *p2;
	int	i, x, npasses, name_found;
	long	pos;

	if (!path) {
		log_message(LOG_ERR, "Missing scheme filename\n");
		return ERROR;
	}
	
	if (!out_info) {
		return ERROR;
	}

	memset(out_info, 0, sizeof(*out_info));
		
	f = fopen(path, "r");
	if (!f) {
		log_message(LOG_ERR, "Could not open scheme file '%s': %s\n", path, strerror(errno));
		return ERROR;
	}

	npasses = 0;
	name_found = 0;
	pos = 0;
	
	while (fgets(buf, sizeof(buf), f)) {
		if (!name_found) {
			// try to find the scheme name
			p1 = strchr(buf, '[');
			p2 = p1 ? strrchr(p1, ']') : NULL;
			if (p1 && p2) {
				*p2 = 0;
				out_info->name = strdup(p1+1);
				pos = ftell(f);
				name_found = 1;
			} else {
				x = sscanf(buf, "%d.", &i);
				if (0 < x) {
					// the scheme does not have a name
					out_info->name = strdup("User-defined");
					name_found = 1;
					
					// Look for pass description
					npasses += wipe_read_pass(buf, sizeof(buf), NULL);
				}
			}
			continue;
		}	    
	
		// Look for pass description
		npasses += wipe_read_pass(buf, sizeof(buf), NULL);
	}


	if (1 > npasses) {
		free(out_info->name);
		fclose(f);
		log_message(LOG_ERR, "No passes defined in '%s' scheme file\n", path);
		return ERROR;
	}
	
	out_info->num_passes = npasses;
	out_info->pass = (pass_s*)malloc(npasses * sizeof(pass_s));
	if (!out_info->pass) {
		free(out_info->name);
		fclose(f);
		log_message(LOG_ERR, "Error allocating memory for a scheme\n");
		return ERROR;
	}
	memset(out_info->pass, 0, npasses * sizeof(pass_s));
	fseek(f, pos, SEEK_SET);

	npasses = 0;

	while (fgets(buf, sizeof(buf), f)) {
		npasses += wipe_read_pass(buf, sizeof(buf), &(out_info->pass[npasses]));
		if (out_info->num_passes == npasses) {
			break;
		}
	}
	
	fclose(f);
		
	return OK;
}



