//===----------------------------------------------------------------------===//
// Copyright © 2025-2026 Apple Inc. and the container project authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//   https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//===----------------------------------------------------------------------===//

import Foundation
import Logging

/// Basic log handler for where simple message output is needed,
/// such as CLI commands.
public struct StderrLogHandler: LogHandler {
    public var logLevel: Logger.Level = .info
    public var metadata: Logger.Metadata = [:]

    public subscript(metadataKey metadataKey: String) -> Logger.Metadata.Value? {
        get {
            self.metadata[metadataKey]
        }
        set {
            self.metadata[metadataKey] = newValue
        }
    }

    public init() {}

    public func log(
        level: Logger.Level,
        message: Logger.Message,
        metadata: Logger.Metadata?,
        source: String,
        file: String,
        function: String,
        line: UInt
    ) {
        let data: Data
        switch logLevel {
        case .debug, .trace:
            let timestamp = ISO8601DateFormatter().string(from: Date())
            if let metadata, !metadata.isEmpty {
                data =
                    "\(timestamp) \(message.description): \(metadata.description)"
                    .data(using: .utf8) ?? Data()
            } else {
                data =
                    "\(timestamp) \(message.description)"
                    .data(using: .utf8) ?? Data()
            }
        default:
            if let metadata, !metadata.isEmpty {
                data =
                    "\(message.description): \(metadata.description)"
                    .data(using: .utf8) ?? Data()
            } else {
                data =
                    message.description
                    .data(using: .utf8) ?? Data()
            }
        }

        // Use a single write call for atomicity
        var output = data
        output.append("\n".data(using: .utf8)!)
        FileHandle.standardError.write(output)
    }
}
