/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { debugLogger, OutputFormat, ExitCodes, getAuthTypeFromEnv, } from '@google/gemini-cli-core';
import { USER_SETTINGS_PATH } from './config/settings.js';
import { validateAuthMethod } from './config/auth.js';
import {} from './config/settings.js';
import { handleError } from './utils/errors.js';
import { runExitCleanup } from './utils/cleanup.js';
export async function validateNonInteractiveAuth(configuredAuthType, useExternalAuth, nonInteractiveConfig, settings) {
    try {
        const effectiveAuthType = configuredAuthType || getAuthTypeFromEnv();
        const enforcedType = settings.merged.security.auth.enforcedType;
        if (enforcedType && effectiveAuthType !== enforcedType) {
            const message = effectiveAuthType
                ? `The enforced authentication type is '${enforcedType}', but the current type is '${effectiveAuthType}'. Please re-authenticate with the correct type.`
                : `The auth type '${enforcedType}' is enforced, but no authentication is configured.`;
            throw new Error(message);
        }
        if (!effectiveAuthType) {
            const message = `Please set an Auth method in your ${USER_SETTINGS_PATH} or specify one of the following environment variables before running: GEMINI_API_KEY, GOOGLE_GENAI_USE_VERTEXAI, GOOGLE_GENAI_USE_GCA`;
            throw new Error(message);
        }
        const authType = effectiveAuthType;
        if (!useExternalAuth) {
            const err = validateAuthMethod(String(authType));
            if (err != null) {
                throw new Error(err);
            }
        }
        return authType;
    }
    catch (error) {
        if (nonInteractiveConfig.getOutputFormat() === OutputFormat.JSON) {
            handleError(error instanceof Error ? error : new Error(String(error)), nonInteractiveConfig, ExitCodes.FATAL_AUTHENTICATION_ERROR);
        }
        else {
            debugLogger.error(error instanceof Error ? error.message : String(error));
            await runExitCleanup();
            process.exit(ExitCodes.FATAL_AUTHENTICATION_ERROR);
        }
    }
}
//# sourceMappingURL=validateNonInterActiveAuth.js.map