!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! This template reads a UV table and writes a LM (2D) cube from it, with
! CUBE capabilities as of 19-MAR-2025.
!
! - CUBE is thought to iterate entries (image or spectrum) mapping
!   leading dimensions (LM or C resp.), and iterating the trailing
!   dimension(s).
! - The real common dimension is the visibility axis. A possibility
!   could be to map the input table dimensions as:
!      C (channel axis) = 1
!      X (vis. axis) = 2
!      Y (fake) = 3
!   and the output image as
!      C (pseudo L) = 1,
!      X (pseudo M) = 2,
!      Y (fake) = 3
!   Then we should be able to use "spectrum" iteration, iterating the
!   trailing X*Y dimensions.
! - However, in practice, the current internal engines actually iterate
!   the Y dimension (i.e. buffer C*X planes) which is degenerate...
! - Because of the above remark, the command is currently coded with
!   full data access, using full-cube types. Two drawbacks:
!     1/ this means we access the 2D data with a 3D type
!     2/ full access means the data must fit in memory all at once.
!   For ease of use, the dimensions must be mapped as:
!     X = 1
!     Y = 2
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeuv_table2image
  use cubetools_parameters
  use cubetools_structure
  use cubetuple_format
  use cube_types
  use cubeadm_cubeid_types
  use cubeadm_cubeprod_types
  use cubeuv_messaging
  !
  public :: table2image
  private
  !
  type :: table2image_comm_t
     type(option_t),     pointer :: comm
     type(cubeid_arg_t), pointer :: uv
     type(cube_prod_t),  pointer :: wei
   contains
     procedure, public  :: register => cubeuv_table2image_comm_register
     procedure, private :: parse    => cubeuv_table2image_comm_parse
     procedure, private :: main     => cubeuv_table2image_comm_main
  end type table2image_comm_t
  type(table2image_comm_t) :: table2image
  !
  type table2image_user_t
     type(cubeid_user_t)     :: cubeids
   contains
     procedure, private :: toprog => cubeuv_table2image_user_toprog
  end type table2image_user_t
  !
  type table2image_prog_t
     class(format_t), pointer :: uv
     type(cube_t),    pointer :: wei
   contains
     procedure, private :: header => cubeuv_table2image_prog_header
     procedure, private :: data   => cubeuv_table2image_prog_data
     procedure, private :: loop   => cubeuv_table2image_prog_loop
     procedure, private :: act    => cubeuv_table2image_prog_act
  end type table2image_prog_t
  !
contains
  !
  subroutine cubeuv_table2image_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(table2image_user_t) :: user
    character(len=*), parameter :: rname='TABLE2IMAGE>COMMAND'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    call table2image%parse(line,user,error)
    if (error) return
    call table2image%main(user,error)
    if (error) continue
  end subroutine cubeuv_table2image_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeuv_table2image_comm_register(comm,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_comm_t), intent(inout) :: comm
    logical,                  intent(inout) :: error
    !
    type(cubeid_arg_t) :: in
    type(cube_prod_t) :: ou
    character(len=*), parameter :: rname='TABLE2IMAGE>COMM>REGISTER'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    ! Syntax
    call cubetools_register_command(&
         'TABLE2IMAGE','[cubeid]',&
         'Transfer the weights or intensities of a UV table in an image',&
         strg_id,&
         cubeuv_table2image_command,&
         comm%comm,&
         error)
    if (error) return
    call in%register(&
         'UVTABLE',&
         'Input UV table',&
         strg_id,&
         code_arg_optional,&
         [flag_uv,flag_table],&
         code_read,&
         code_access_fullset,&
         comm%uv,&
         error)
    if (error) return
    !
    ! Products
    call ou%register(&
         'IMAGE',&
         'Output image',&
         strg_id,&
         [flag_weight,flag_image],&
         comm%wei,&
         error)
    if (error)  return
  end subroutine cubeuv_table2image_comm_register
  !
  subroutine cubeuv_table2image_comm_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    ! table2image uvid
    !----------------------------------------------------------------------
    class(table2image_comm_t), intent(in)    :: comm
    character(len=*),          intent(in)    :: line
    type(table2image_user_t),  intent(out)   :: user
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='TABLE2IMAGE>COMM>PARSE'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,comm%comm,user%cubeids,error)
    if (error) return
  end subroutine cubeuv_table2image_comm_parse
  !
  subroutine cubeuv_table2image_comm_main(comm,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_comm_t), intent(in)    :: comm
    type(table2image_user_t),  intent(inout) :: user
    logical,                   intent(inout) :: error
    !
    type(table2image_prog_t) :: prog
    character(len=*), parameter :: rname='TABLE2IMAGE>MAIN'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(comm,error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubeuv_table2image_comm_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeuv_table2image_user_toprog(user,comm,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_user_t), intent(in)    :: user
    type(table2image_comm_t),  intent(in)    :: comm
    type(table2image_prog_t),  intent(out)   :: prog
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='TABLE2IMAGE>USER>TOPROG'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    call cubeadm_get_fheader(comm%uv,user%cubeids,prog%uv,error)
    if (error) return
    ! User feedback about the interpretation of his command line
  end subroutine cubeuv_table2image_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubeuv_table2image_prog_header(prog,comm,error)
    use cubetools_axis_types
    use cubetools_unit_setup
    use cubeadm_clone
    use cubetools_header_methods
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_prog_t), intent(inout) :: prog
    type(table2image_comm_t),  intent(in)    :: comm
    logical,                   intent(inout) :: error
    !
    type(axis_t) :: axis
    character(len=*), parameter :: rname='TABLE2IMAGE>PROG>HEADER'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    if (prog%uv%head%set%il.ne.1 .or.  &
        prog%uv%head%set%im.ne.2) then
      call cubeuv_message(seve%e,rname,  &
        'The table dimensions must have been remapped as IX=1 and IY=2')
      error = .true.
      return
    endif
    !
    call cubeadm_clone_header(comm%wei,prog%uv,prog%wei,error)
    if (error) return
    !
    ! 1st axis: from the UV table frequency axis
    call cubetools_header_get_axis_head_l(prog%wei%head,axis,error)
    if (error) return
    axis%name = 'Channels'
    axis%unit = ''
    axis%kind = code_unit_fov  ! ZZZ
    axis%genuine = .true.
    axis%regular = .true.
    ! Assume 7 leading DAPS and 3 values per atom (the header should be
    ! enriched to deal with these values)
    axis%n   = (prog%uv%head%set%axis(prog%uv%head%set%il)%n-7)/3
    axis%ref = 1.d0
    axis%val = 1.d0
    axis%inc = 1.d0
    call cubetools_header_update_axset_l(axis,prog%wei%head,error)
    if (error)  return
    !
    ! 2nd axis: from the UV table visi axis
    call cubetools_header_get_axis_head_m(prog%wei%head,axis,error)
    if (error) return
    axis%name = 'Visibilities'
    axis%unit = ''
    axis%kind = code_unit_fov  ! ZZZ
    axis%genuine = .true.
    axis%regular = .true.
    axis%n   = prog%uv%head%set%axis(prog%uv%head%set%im)%n
    axis%ref = 1.d0
    axis%val = 1.d0
    axis%inc = 1.d0
    call cubetools_header_update_axset_m(axis,prog%wei%head,error)
    if (error)  return
    !
    ! 3rd axis: none
    call cubetools_header_nullify_axset_c(prog%wei%head,error)
    if (error)  return
  end subroutine cubeuv_table2image_prog_header
  !
  subroutine cubeuv_table2image_prog_data(prog,error)
    use cubeadm_opened
    use cubeadm_fullcube_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_prog_t), intent(inout) :: prog
    logical,                   intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    type(fullcube_t) :: uv,wei
    character(len=*), parameter :: rname='TABLE2IMAGE>PROG>DATA'
    !
    call cubeuv_message(uvseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !
    call uv%associate('uv table',prog%uv,error)
    if (error) return
    call wei%allocate('weight image',prog%wei,error)
    if (error) return
    !
    ! Loop on task => Only one task, (i.e. one iteration) => no parallelization statements
    do while (cubeadm_dataiterate_all(iter,error))
       if (error) exit
       if (.not.error) &
         call prog%loop(iter,uv,wei,error)
    enddo ! itask
  end subroutine cubeuv_table2image_prog_data
  !
  subroutine cubeuv_table2image_prog_loop(prog,iter,uv,wei,error)
    use cubeadm_taskloop
    use cubeadm_fullcube_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_prog_t), intent(inout) :: prog
    type(cubeadm_iterator_t),  intent(inout) :: iter
    type(fullcube_t),          intent(inout) :: uv
    type(fullcube_t),          intent(inout) :: wei
    logical,                   intent(inout) :: error
    !
    character(len=*), parameter :: rname='TABLE2IMAGE>PROG>LOOP'
    !
    do while (iter%iterate_entry(error))  ! Only 1 iteration in this case
      ! Get the uv table
      call uv%get(error)
      if (error) return
      ! Transfer
      call prog%act(uv,wei,error)
      if (error) return
      ! Write the weight image
      call wei%put(error)
      if (error) return
    enddo  ! ientry
  end subroutine cubeuv_table2image_prog_loop
  !
  subroutine cubeuv_table2image_prog_act(prog,uv,wei,error)
    use cubeadm_fullcube_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(table2image_prog_t), intent(inout) :: prog
    type(fullcube_t),          intent(inout) :: uv
    type(fullcube_t),          intent(inout) :: wei
    logical,                   intent(inout) :: error
    !
    integer(kind=indx_k), parameter :: one=1
    integer(kind=indx_k) :: ix,jx
    integer(kind=indx_k) :: iy
    character(len=*), parameter :: rname='TABLE2IMAGE>PROG>ACT'
    !
    ! Loop on visibilities (2nd dimension)
    do iy=1,wei%ny
       ! Loop on channel (1st dimension)
       do ix=1,wei%nx
          jx = 7+ix*3
          wei%val(ix,iy,one) = uv%val(jx,iy,one)
       enddo ! ox
    enddo ! oy
  end subroutine cubeuv_table2image_prog_act
end module cubeuv_table2image
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
