---
title: "Generating Secrets"
linkTitle: "Generating Secrets"
weight: 2
date: 2023-10-20
description: >
    Working with the Secret Generator
---

Secret objects can be generated by adding a [`secretGenerator`] entry to the `kustomization.yaml` file. This is similar to the [`configMapGenerator`]. Secret Resources may be generated from files and literals. It is important to note that the secrets are base64 encoded.

## Create Secret from a file

To generate a Secret Resource from a file, add an entry to [`secretGenerator`] with the filename.

The Secret will have data values populated from the file contents. The contents of each file will appear as a single data item in the Secret keyed by the filename.

The following example generates a Secret with a data item containing the contents of a file.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
secretGenerator:
- name: db-user-pass
  files:
  - credentials.txt
```

2. Create a `credentials.txt` file.
```yaml
# credentials.txt
username=admin
password=S!B\*d$zDsb=
```

3. Create the Secret using `kustomize build`.
```bash
kustomize build .
```

The Secret manifest is generated.
```yaml
apiVersion: v1
kind: Secret
type: Opaque
metadata:
  name: db-user-pass-gf9bgh225c
data:
  credentials.txt: dXNlcm5hbWU9YWRtaW4KcGFzc3dvcmQ9UyFCXCpkJHpEc2I9Cg==
```

The credentials key value is base64 encoded.
```bash
echo "dXNlcm5hbWU9YWRtaW4KcGFzc3dvcmQ9UyFCXCpkJHpEc2I9Cg==" | base64 -d
username=admin
password=S!B\*d$zDsb=
```

## Create Secret from literals

To generate a Secret Resource from literal key-value pairs, add an entry to [`secretGenerator`] with a
list of `literals`.

{{< alert color="success" title="Literal Syntax" >}}
- The key/value are separated by a `=` sign (left side is the key).
- The value of each literal will appear as a data item in the Secret keyed by its key.
{{< /alert >}}

The following example generates a Secret with two data items generated from literals.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
secretGenerator:
- name: db-user-pass
  literals:
    - username=admin
    - password=S!B\*d$zDsb=
```

2. Create the Secret using `kustomize build`.
```bash
kustomize build .
```

The Secret manifest is generated.
```yaml
apiVersion: v1
kind: Secret
type: Opaque
metadata:
  name: db-user-pass-t8d2d65755
data:
  password: UyFCXCpkJHpEc2I9
  username: YWRtaW4=
```

The credential key values are base64 encoded.
```bash
echo "UyFCXCpkJHpEc2I9" | base64 -d
S!B\*d$zDsb=

echo "YWRtaW4=" | base64 -d
admin
```

## Create a TLS Secret

The following example generates a TLS Secret with certificate and private key data files.

1. Create a Kustomization file.
```yaml
# kustomization.yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization
secretGenerator:
- name: app-tls
  files:
    - "tls.crt"
    - "tls.key"
  type: "kubernetes.io/tls"
```

2. Create a certificate file.
```yaml
# tls.crt
LS0tLS1CRUd...tCg==
```

3. Create a private key file.
```yaml
# tls.key
LS0tLS1CRUd...0tLQo=
```

4. Create the Secret using `kustomize build`.
```bash
kustomize build .
```

The Secret manifest is generated. The data key values are base64 encoded.
```yaml
apiVersion: v1
kind: Secret
type: kubernetes.io/tls
metadata:
  name: app-tls-c888dfbhf8
data:
  tls.crt: TFMwdExTMUNSVWQuLi50Q2c9PQ==
  tls.key: TFMwdExTMUNSVWQuLi4wdExRbz0=
```

[`secretGenerator`]: /docs/reference/api/kustomization-file/secretgenerator/
[`configMapGenerator`]: /docs/reference/api/kustomization-file/configmapgenerator/
