#!/bin/sh

set -e
cd "$(dirname $0)"

# recreate testrepo and external
rm -r testrepo external

# build base images
for i in 1 2 3; do
  docker buildx build \
    -t "testrepo:b${i}" -o "type=oci,dest=b${i}.tar" \
    --platform "linux/amd64,linux/arm64,linux/arm/v7,linux/arm/v6" \
    --provenance=false \
    -f "Dockerfile.b${i}" .
  regctl image import "ocidir://testrepo:b${i}" "b${i}.tar"
  regctl image mod \
    --annotation "org.example.version=b${i}" \
    --time-max 2020-01-01T00:00:00Z \
    --replace "ocidir://testrepo:b${i}"
  regctl image export "ocidir://testrepo:b${i}" "b${i}.tar"
done

# build 3 images
# use oci-layout after next release of docker enables it
# the hacky workaround is to use busybox and do a rebase in the image mod
#  --build-context testrepo:b1=oci-layout://b1.tar \
docker buildx build \
  -t "testrepo:v1" -o "type=oci,dest=v1.tar" \
  --build-context testrepo:b1=docker-image://busybox:latest \
  --build-arg "arg=build-v1" --build-arg "arg_label=arg_for_label" \
  --platform "linux/amd64,linux/arm64" \
  --provenance=false \
  -f "Dockerfile.v1" .
docker buildx build \
  -t "testrepo:v2" -o "type=oci,dest=v2.tar" \
  --build-context testrepo:b1=docker-image://busybox:latest \
  --build-arg "arg=build-v2" --build-arg "arg_label=arg_for_label" \
  --platform "linux/amd64,linux/arm64,linux/arm/v7" \
  --provenance=false \
  -f "Dockerfile.v2" .
docker buildx build \
  -t "testrepo:v3" -o "type=oci,dest=v3.tar" \
  --build-context testrepo:b1=docker-image://busybox:latest \
  --build-arg "arg=build-v3" --build-arg "arg_label=arg_for_label" \
  --platform "linux/amd64,linux/arm64,linux/arm/v7,linux/arm/v6" \
  --provenance=false \
  -f "Dockerfile.v3" .

# add 3 images
for i in 1 2 3; do
  regctl image import "ocidir://testrepo:v${i}" "v${i}.tar"
  regctl image mod \
    --rebase-ref "busybox,ocidir://testrepo:b1" \
    --annotation "org.example.version=v${i}" \
    --time-max 2021-01-01T00:00:00Z \
    --replace "ocidir://testrepo:v${i}"
  rm "v${i}.tar"
done

# set the base image annotations
for i in 2 3; do
  regctl image mod \
    --annotation "org.opencontainers.image.base.name=registry.example.org/testrepo:b${i}" \
    --annotation "org.opencontainers.image.base.digest=$(regctl image digest ocidir://testrepo:b1)" \
    --replace "ocidir://testrepo:v${i}"
done

# create docker-like attestations on v1
echo scripted build | regctl artifact put \
  --media-type application/vnd.oci.image.manifest.v1+json \
  --config-type application/vnd.oci.image.config.v1+json \
  --file-media-type application/vnd.example.build-type+json \
  ocidir://testrepo:a-docker
echo scripted oci build | regctl artifact put \
  --media-type application/vnd.oci.image.manifest.v1+json \
  --config-type application/vnd.oci.image.config.v1+json \
  --file-media-type application/vnd.example.build-type+json \
  --subject ocidir://testrepo:v1 --platform linux/arm64 \
  ocidir://testrepo:a-docker-oci
regctl index add \
  --ref "ocidir://testrepo:a-docker" \
  --desc-annotation "vnd.docker.reference.type=builder" \
  --desc-annotation "vnd.docker.reference.digest=$(regctl image digest --platform linux/amd64 ocidir://testrepo:v1)" \
  --desc-platform "unknown/unknown" \
  ocidir://testrepo:v1
regctl index add \
  --ref "ocidir://testrepo:a-docker-oci" \
  --desc-annotation "vnd.docker.reference.type=builder" \
  --desc-annotation "vnd.docker.reference.digest=$(regctl image digest --platform linux/arm64 ocidir://testrepo:v1)" \
  --desc-platform "unknown/unknown" \
  ocidir://testrepo:v1

# create two artifacts on v2
echo eggs | regctl artifact put \
  --artifact-type application/example.sbom -m application/example.sbom.breakfast \
  --subject ocidir://testrepo:v2 ocidir://testrepo:a1
echo signed | regctl artifact put \
  --artifact-type application/example.signature -m application/example.signature.text \
  --subject ocidir://testrepo:v2 ocidir://testrepo:a2
echo no arms | regctl artifact put \
  --artifact-type application/example.arms -m application/example.arms \
  --subject ocidir://testrepo:v2 --platform linux/amd64
echo no arms | regctl artifact put \
  --artifact-type application/example.arms -m application/example.arms \
  --subject ocidir://testrepo:v2 --platform linux/amd64
echo 7 arms | regctl artifact put \
  --artifact-type application/example.arms -m application/example.arms \
  --subject ocidir://testrepo:v2 --platform linux/arm/v7
echo 64 arms | regctl artifact put \
  --artifact-type application/example.arms -m application/example.arms \
  --subject ocidir://testrepo:v2 --platform linux/arm64

# another standalone artifact with multiple layers
regctl artifact put \
  --artifact-type application/example.layers \
  -m application/example.layer.1 --file-title -f layer1.txt \
  -m application/example.layer.2 --file-title -f layer2.txt \
  -m application/example.layer.3 --file-title -f layer3.txt \
  ocidir://testrepo:a3

# include a external artifacts
echo "bacon" | regctl artifact put \
  --artifact-type application/example.sbom -m application/example.sbom.breakfast \
  --annotation preference=1 \
  --subject ocidir://testrepo:v2 ocidir://external:a3
echo "ham" | regctl artifact put \
  --artifact-type application/example.sbom -m application/example.sbom.breakfast \
  --annotation preference=2 \
  --subject ocidir://testrepo:v2 ocidir://external:a4

# create an index of artifacts
regctl index create \
  --artifact-type application/example.artifacts \
  --ref ocidir://testrepo:a1 \
  --desc-annotation "type=sbom" \
  ocidir://testrepo:ai
regctl index add \
  --ref ocidir://testrepo:a2 \
  --desc-annotation "type=sig" \
  ocidir://testrepo:ai

# create a standalone artifact with a data field on the config
echo example | regctl artifact put \
  --artifact-type application/example -m application/example \
  ocidir://testrepo:a-example
regctl image mod --data-max 3 --replace ocidir://testrepo:a-example

# create a digest tag from v3 pointing to v1
v1_dig="$(regctl image digest ocidir://testrepo:v1)"
v1_dig=${v1_dig#*:}
v3_dig="$(regctl image digest ocidir://testrepo:v3)"
v3_shortdig="$(echo ${v3_dig#*:} | cut -b1-16)"
regctl image copy ocidir://testrepo:v3 "ocidir://testrepo:sha256-${v1_dig}.${v3_shortdig}.meta"

# create a looping index that references a child in both the manifests and subject
echo "child" | regctl artifact put --artifact-type application/example.child ocidir://testrepo:child
regctl index create ocidir://testrepo:loop \
  --artifact-type application/example.loop \
  --subject child \
  --ref ocidir://testrepo:child

# create an artifact with a digest tag to itself
echo mirror | regctl artifact put \
  --artifact-type application/example.mirror \
  ocidir://testrepo:mirror
regctl image copy ocidir://testrepo:mirror \
  "ocidir://testrepo:$(regctl manifest head ocidir://testrepo:mirror --format '{{.GetDescriptor.Digest.Algorithm}}-{{.GetDescriptor.Digest.Hex}}')"

rm b1.tar b2.tar b3.tar
