# -*- coding: utf-8 -*- #
# Copyright 2026 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Implementation of gcloud kms retired-resources list command.

This command is used to list all retired resources within a specified KMS
location.
"""

import argparse
from collections.abc import Iterator

from apitools.base.py import list_pager
from googlecloudsdk.api_lib.cloudkms import base as cloudkms_base
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.kms import resource_args
from googlecloudsdk.generated_clients.apis.cloudkms.v1 import cloudkms_v1_messages


@base.DefaultUniverseOnly
class List(base.ListCommand):
  """List the retired resources within a location.

  Lists all retired resources within the given location.

  ## EXAMPLES

  The following command lists a maximum of five retired resources in the
  location
  `global`:

    $ {command} --location=global --limit=5
  """

  @classmethod
  def Args(cls, parser: argparse.ArgumentParser) -> None:
    resource_args.AddKmsLocationResourceArgForKMS(parser, True, '--location')
    parser.display_info.AddFormat('table(name)')

  def Run(
      self, args: argparse.Namespace
  ) -> Iterator[cloudkms_v1_messages.RetiredResource]:
    client = cloudkms_base.GetClientInstance()
    messages = cloudkms_base.GetMessagesModule()

    location_ref = args.CONCEPTS.location.Parse()

    return list_pager.YieldFromList(
        client.projects_locations_retiredResources,
        messages.CloudkmsProjectsLocationsRetiredResourcesListRequest(
            parent=location_ref.RelativeName()
        ),
        field='retiredResources',
        limit=args.limit,
        batch_size_attribute='pageSize',
    )
