/*
 * Hurl (https://hurl.dev)
 * Copyright (C) 2025 Orange
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *          http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
use std::{collections::HashMap, sync::LazyLock};

// HTML5 named character references
//
// Generated from <https://html.spec.whatwg.org/entities.json> and
// <https://html.spec.whatwg.org/multipage/named-characters.html>.
// Map HTML5 named character references to the equivalent Unicode character(s).
static HTML5_ENTITIES: [(&str, &str); 2231] = [
    ("AElig", "\u{C6}"),
    ("AElig;", "\u{C6}"),
    ("AMP", "\u{26}"),
    ("AMP;", "\u{26}"),
    ("Aacute", "\u{C1}"),
    ("Aacute;", "\u{C1}"),
    ("Abreve;", "\u{102}"),
    ("Acirc", "\u{C2}"),
    ("Acirc;", "\u{C2}"),
    ("Acy;", "\u{410}"),
    ("Afr;", "\u{1D504}"),
    ("Agrave", "\u{C0}"),
    ("Agrave;", "\u{C0}"),
    ("Alpha;", "\u{391}"),
    ("Amacr;", "\u{100}"),
    ("And;", "\u{2A53}"),
    ("Aogon;", "\u{104}"),
    ("Aopf;", "\u{1D538}"),
    ("ApplyFunction;", "\u{2061}"),
    ("Aring", "\u{C5}"),
    ("Aring;", "\u{C5}"),
    ("Ascr;", "\u{1D49C}"),
    ("Assign;", "\u{2254}"),
    ("Atilde", "\u{C3}"),
    ("Atilde;", "\u{C3}"),
    ("Auml", "\u{C4}"),
    ("Auml;", "\u{C4}"),
    ("Backslash;", "\u{2216}"),
    ("Barv;", "\u{2AE7}"),
    ("Barwed;", "\u{2306}"),
    ("Bcy;", "\u{411}"),
    ("Because;", "\u{2235}"),
    ("Bernoullis;", "\u{212C}"),
    ("Beta;", "\u{392}"),
    ("Bfr;", "\u{1D505}"),
    ("Bopf;", "\u{1D539}"),
    ("Breve;", "\u{2D8}"),
    ("Bscr;", "\u{212C}"),
    ("Bumpeq;", "\u{224E}"),
    ("CHcy;", "\u{427}"),
    ("COPY", "\u{A9}"),
    ("COPY;", "\u{A9}"),
    ("Cacute;", "\u{106}"),
    ("Cap;", "\u{22D2}"),
    ("CapitalDifferentialD;", "\u{2145}"),
    ("Cayleys;", "\u{212D}"),
    ("Ccaron;", "\u{10C}"),
    ("Ccedil", "\u{C7}"),
    ("Ccedil;", "\u{C7}"),
    ("Ccirc;", "\u{108}"),
    ("Cconint;", "\u{2230}"),
    ("Cdot;", "\u{10A}"),
    ("Cedilla;", "\u{B8}"),
    ("CenterDot;", "\u{B7}"),
    ("Cfr;", "\u{212D}"),
    ("Chi;", "\u{3A7}"),
    ("CircleDot;", "\u{2299}"),
    ("CircleMinus;", "\u{2296}"),
    ("CirclePlus;", "\u{2295}"),
    ("CircleTimes;", "\u{2297}"),
    ("ClockwiseContourIntegral;", "\u{2232}"),
    ("CloseCurlyDoubleQuote;", "\u{201D}"),
    ("CloseCurlyQuote;", "\u{2019}"),
    ("Colon;", "\u{2237}"),
    ("Colone;", "\u{2A74}"),
    ("Congruent;", "\u{2261}"),
    ("Conint;", "\u{222F}"),
    ("ContourIntegral;", "\u{222E}"),
    ("Copf;", "\u{2102}"),
    ("Coproduct;", "\u{2210}"),
    ("CounterClockwiseContourIntegral;", "\u{2233}"),
    ("Cross;", "\u{2A2F}"),
    ("Cscr;", "\u{1D49E}"),
    ("Cup;", "\u{22D3}"),
    ("CupCap;", "\u{224D}"),
    ("DD;", "\u{2145}"),
    ("DDotrahd;", "\u{2911}"),
    ("DJcy;", "\u{402}"),
    ("DScy;", "\u{405}"),
    ("DZcy;", "\u{40F}"),
    ("Dagger;", "\u{2021}"),
    ("Darr;", "\u{21A1}"),
    ("Dashv;", "\u{2AE4}"),
    ("Dcaron;", "\u{10E}"),
    ("Dcy;", "\u{414}"),
    ("Del;", "\u{2207}"),
    ("Delta;", "\u{394}"),
    ("Dfr;", "\u{1D507}"),
    ("DiacriticalAcute;", "\u{B4}"),
    ("DiacriticalDot;", "\u{2D9}"),
    ("DiacriticalDoubleAcute;", "\u{2DD}"),
    ("DiacriticalGrave;", "\u{60}"),
    ("DiacriticalTilde;", "\u{2DC}"),
    ("Diamond;", "\u{22C4}"),
    ("DifferentialD;", "\u{2146}"),
    ("Dopf;", "\u{1D53B}"),
    ("Dot;", "\u{A8}"),
    ("DotDot;", "\u{20DC}"),
    ("DotEqual;", "\u{2250}"),
    ("DoubleContourIntegral;", "\u{222F}"),
    ("DoubleDot;", "\u{A8}"),
    ("DoubleDownArrow;", "\u{21D3}"),
    ("DoubleLeftArrow;", "\u{21D0}"),
    ("DoubleLeftRightArrow;", "\u{21D4}"),
    ("DoubleLeftTee;", "\u{2AE4}"),
    ("DoubleLongLeftArrow;", "\u{27F8}"),
    ("DoubleLongLeftRightArrow;", "\u{27FA}"),
    ("DoubleLongRightArrow;", "\u{27F9}"),
    ("DoubleRightArrow;", "\u{21D2}"),
    ("DoubleRightTee;", "\u{22A8}"),
    ("DoubleUpArrow;", "\u{21D1}"),
    ("DoubleUpDownArrow;", "\u{21D5}"),
    ("DoubleVerticalBar;", "\u{2225}"),
    ("DownArrow;", "\u{2193}"),
    ("DownArrowBar;", "\u{2913}"),
    ("DownArrowUpArrow;", "\u{21F5}"),
    ("DownBreve;", "\u{311}"),
    ("DownLeftRightVector;", "\u{2950}"),
    ("DownLeftTeeVector;", "\u{295E}"),
    ("DownLeftVector;", "\u{21BD}"),
    ("DownLeftVectorBar;", "\u{2956}"),
    ("DownRightTeeVector;", "\u{295F}"),
    ("DownRightVector;", "\u{21C1}"),
    ("DownRightVectorBar;", "\u{2957}"),
    ("DownTee;", "\u{22A4}"),
    ("DownTeeArrow;", "\u{21A7}"),
    ("Downarrow;", "\u{21D3}"),
    ("Dscr;", "\u{1D49F}"),
    ("Dstrok;", "\u{110}"),
    ("ENG;", "\u{14A}"),
    ("ETH", "\u{D0}"),
    ("ETH;", "\u{D0}"),
    ("Eacute", "\u{C9}"),
    ("Eacute;", "\u{C9}"),
    ("Ecaron;", "\u{11A}"),
    ("Ecirc", "\u{CA}"),
    ("Ecirc;", "\u{CA}"),
    ("Ecy;", "\u{42D}"),
    ("Edot;", "\u{116}"),
    ("Efr;", "\u{1D508}"),
    ("Egrave", "\u{C8}"),
    ("Egrave;", "\u{C8}"),
    ("Element;", "\u{2208}"),
    ("Emacr;", "\u{112}"),
    ("EmptySmallSquare;", "\u{25FB}"),
    ("EmptyVerySmallSquare;", "\u{25AB}"),
    ("Eogon;", "\u{118}"),
    ("Eopf;", "\u{1D53C}"),
    ("Epsilon;", "\u{395}"),
    ("Equal;", "\u{2A75}"),
    ("EqualTilde;", "\u{2242}"),
    ("Equilibrium;", "\u{21CC}"),
    ("Escr;", "\u{2130}"),
    ("Esim;", "\u{2A73}"),
    ("Eta;", "\u{397}"),
    ("Euml", "\u{CB}"),
    ("Euml;", "\u{CB}"),
    ("Exists;", "\u{2203}"),
    ("ExponentialE;", "\u{2147}"),
    ("Fcy;", "\u{424}"),
    ("Ffr;", "\u{1D509}"),
    ("FilledSmallSquare;", "\u{25FC}"),
    ("FilledVerySmallSquare;", "\u{25AA}"),
    ("Fopf;", "\u{1D53D}"),
    ("ForAll;", "\u{2200}"),
    ("Fouriertrf;", "\u{2131}"),
    ("Fscr;", "\u{2131}"),
    ("GJcy;", "\u{403}"),
    ("GT", "\u{3E}"),
    ("GT;", "\u{3E}"),
    ("Gamma;", "\u{393}"),
    ("Gammad;", "\u{3DC}"),
    ("Gbreve;", "\u{11E}"),
    ("Gcedil;", "\u{122}"),
    ("Gcirc;", "\u{11C}"),
    ("Gcy;", "\u{413}"),
    ("Gdot;", "\u{120}"),
    ("Gfr;", "\u{1D50A}"),
    ("Gg;", "\u{22D9}"),
    ("Gopf;", "\u{1D53E}"),
    ("GreaterEqual;", "\u{2265}"),
    ("GreaterEqualLess;", "\u{22DB}"),
    ("GreaterFullEqual;", "\u{2267}"),
    ("GreaterGreater;", "\u{2AA2}"),
    ("GreaterLess;", "\u{2277}"),
    ("GreaterSlantEqual;", "\u{2A7E}"),
    ("GreaterTilde;", "\u{2273}"),
    ("Gscr;", "\u{1D4A2}"),
    ("Gt;", "\u{226B}"),
    ("HARDcy;", "\u{42A}"),
    ("Hacek;", "\u{2C7}"),
    ("Hat;", "\u{5E}"),
    ("Hcirc;", "\u{124}"),
    ("Hfr;", "\u{210C}"),
    ("HilbertSpace;", "\u{210B}"),
    ("Hopf;", "\u{210D}"),
    ("HorizontalLine;", "\u{2500}"),
    ("Hscr;", "\u{210B}"),
    ("Hstrok;", "\u{126}"),
    ("HumpDownHump;", "\u{224E}"),
    ("HumpEqual;", "\u{224F}"),
    ("IEcy;", "\u{415}"),
    ("IJlig;", "\u{132}"),
    ("IOcy;", "\u{401}"),
    ("Iacute", "\u{CD}"),
    ("Iacute;", "\u{CD}"),
    ("Icirc", "\u{CE}"),
    ("Icirc;", "\u{CE}"),
    ("Icy;", "\u{418}"),
    ("Idot;", "\u{130}"),
    ("Ifr;", "\u{2111}"),
    ("Igrave", "\u{CC}"),
    ("Igrave;", "\u{CC}"),
    ("Im;", "\u{2111}"),
    ("Imacr;", "\u{12A}"),
    ("ImaginaryI;", "\u{2148}"),
    ("Implies;", "\u{21D2}"),
    ("Int;", "\u{222C}"),
    ("Integral;", "\u{222B}"),
    ("Intersection;", "\u{22C2}"),
    ("InvisibleComma;", "\u{2063}"),
    ("InvisibleTimes;", "\u{2062}"),
    ("Iogon;", "\u{12E}"),
    ("Iopf;", "\u{1D540}"),
    ("Iota;", "\u{399}"),
    ("Iscr;", "\u{2110}"),
    ("Itilde;", "\u{128}"),
    ("Iukcy;", "\u{406}"),
    ("Iuml", "\u{CF}"),
    ("Iuml;", "\u{CF}"),
    ("Jcirc;", "\u{134}"),
    ("Jcy;", "\u{419}"),
    ("Jfr;", "\u{1D50D}"),
    ("Jopf;", "\u{1D541}"),
    ("Jscr;", "\u{1D4A5}"),
    ("Jsercy;", "\u{408}"),
    ("Jukcy;", "\u{404}"),
    ("KHcy;", "\u{425}"),
    ("KJcy;", "\u{40C}"),
    ("Kappa;", "\u{39A}"),
    ("Kcedil;", "\u{136}"),
    ("Kcy;", "\u{41A}"),
    ("Kfr;", "\u{1D50E}"),
    ("Kopf;", "\u{1D542}"),
    ("Kscr;", "\u{1D4A6}"),
    ("LJcy;", "\u{409}"),
    ("LT", "\u{3C}"),
    ("LT;", "\u{3C}"),
    ("Lacute;", "\u{139}"),
    ("Lambda;", "\u{39B}"),
    ("Lang;", "\u{27EA}"),
    ("Laplacetrf;", "\u{2112}"),
    ("Larr;", "\u{219E}"),
    ("Lcaron;", "\u{13D}"),
    ("Lcedil;", "\u{13B}"),
    ("Lcy;", "\u{41B}"),
    ("LeftAngleBracket;", "\u{27E8}"),
    ("LeftArrow;", "\u{2190}"),
    ("LeftArrowBar;", "\u{21E4}"),
    ("LeftArrowRightArrow;", "\u{21C6}"),
    ("LeftCeiling;", "\u{2308}"),
    ("LeftDoubleBracket;", "\u{27E6}"),
    ("LeftDownTeeVector;", "\u{2961}"),
    ("LeftDownVector;", "\u{21C3}"),
    ("LeftDownVectorBar;", "\u{2959}"),
    ("LeftFloor;", "\u{230A}"),
    ("LeftRightArrow;", "\u{2194}"),
    ("LeftRightVector;", "\u{294E}"),
    ("LeftTee;", "\u{22A3}"),
    ("LeftTeeArrow;", "\u{21A4}"),
    ("LeftTeeVector;", "\u{295A}"),
    ("LeftTriangle;", "\u{22B2}"),
    ("LeftTriangleBar;", "\u{29CF}"),
    ("LeftTriangleEqual;", "\u{22B4}"),
    ("LeftUpDownVector;", "\u{2951}"),
    ("LeftUpTeeVector;", "\u{2960}"),
    ("LeftUpVector;", "\u{21BF}"),
    ("LeftUpVectorBar;", "\u{2958}"),
    ("LeftVector;", "\u{21BC}"),
    ("LeftVectorBar;", "\u{2952}"),
    ("Leftarrow;", "\u{21D0}"),
    ("Leftrightarrow;", "\u{21D4}"),
    ("LessEqualGreater;", "\u{22DA}"),
    ("LessFullEqual;", "\u{2266}"),
    ("LessGreater;", "\u{2276}"),
    ("LessLess;", "\u{2AA1}"),
    ("LessSlantEqual;", "\u{2A7D}"),
    ("LessTilde;", "\u{2272}"),
    ("Lfr;", "\u{1D50F}"),
    ("Ll;", "\u{22D8}"),
    ("Lleftarrow;", "\u{21DA}"),
    ("Lmidot;", "\u{13F}"),
    ("LongLeftArrow;", "\u{27F5}"),
    ("LongLeftRightArrow;", "\u{27F7}"),
    ("LongRightArrow;", "\u{27F6}"),
    ("Longleftarrow;", "\u{27F8}"),
    ("Longleftrightarrow;", "\u{27FA}"),
    ("Longrightarrow;", "\u{27F9}"),
    ("Lopf;", "\u{1D543}"),
    ("LowerLeftArrow;", "\u{2199}"),
    ("LowerRightArrow;", "\u{2198}"),
    ("Lscr;", "\u{2112}"),
    ("Lsh;", "\u{21B0}"),
    ("Lstrok;", "\u{141}"),
    ("Lt;", "\u{226A}"),
    ("Map;", "\u{2905}"),
    ("Mcy;", "\u{41C}"),
    ("MediumSpace;", "\u{205F}"),
    ("Mellintrf;", "\u{2133}"),
    ("Mfr;", "\u{1D510}"),
    ("MinusPlus;", "\u{2213}"),
    ("Mopf;", "\u{1D544}"),
    ("Mscr;", "\u{2133}"),
    ("Mu;", "\u{39C}"),
    ("NJcy;", "\u{40A}"),
    ("Nacute;", "\u{143}"),
    ("Ncaron;", "\u{147}"),
    ("Ncedil;", "\u{145}"),
    ("Ncy;", "\u{41D}"),
    ("NegativeMediumSpace;", "\u{200B}"),
    ("NegativeThickSpace;", "\u{200B}"),
    ("NegativeThinSpace;", "\u{200B}"),
    ("NegativeVeryThinSpace;", "\u{200B}"),
    ("NestedGreaterGreater;", "\u{226B}"),
    ("NestedLessLess;", "\u{226A}"),
    ("NewLine;", "\u{A}"),
    ("Nfr;", "\u{1D511}"),
    ("NoBreak;", "\u{2060}"),
    ("NonBreakingSpace;", "\u{A0}"),
    ("Nopf;", "\u{2115}"),
    ("Not;", "\u{2AEC}"),
    ("NotCongruent;", "\u{2262}"),
    ("NotCupCap;", "\u{226D}"),
    ("NotDoubleVerticalBar;", "\u{2226}"),
    ("NotElement;", "\u{2209}"),
    ("NotEqual;", "\u{2260}"),
    ("NotEqualTilde;", "\u{2242}\u{338}"),
    ("NotExists;", "\u{2204}"),
    ("NotGreater;", "\u{226F}"),
    ("NotGreaterEqual;", "\u{2271}"),
    ("NotGreaterFullEqual;", "\u{2267}\u{338}"),
    ("NotGreaterGreater;", "\u{226B}\u{338}"),
    ("NotGreaterLess;", "\u{2279}"),
    ("NotGreaterSlantEqual;", "\u{2A7E}\u{338}"),
    ("NotGreaterTilde;", "\u{2275}"),
    ("NotHumpDownHump;", "\u{224E}\u{338}"),
    ("NotHumpEqual;", "\u{224F}\u{338}"),
    ("NotLeftTriangle;", "\u{22EA}"),
    ("NotLeftTriangleBar;", "\u{29CF}\u{338}"),
    ("NotLeftTriangleEqual;", "\u{22EC}"),
    ("NotLess;", "\u{226E}"),
    ("NotLessEqual;", "\u{2270}"),
    ("NotLessGreater;", "\u{2278}"),
    ("NotLessLess;", "\u{226A}\u{338}"),
    ("NotLessSlantEqual;", "\u{2A7D}\u{338}"),
    ("NotLessTilde;", "\u{2274}"),
    ("NotNestedGreaterGreater;", "\u{2AA2}\u{338}"),
    ("NotNestedLessLess;", "\u{2AA1}\u{338}"),
    ("NotPrecedes;", "\u{2280}"),
    ("NotPrecedesEqual;", "\u{2AAF}\u{338}"),
    ("NotPrecedesSlantEqual;", "\u{22E0}"),
    ("NotReverseElement;", "\u{220C}"),
    ("NotRightTriangle;", "\u{22EB}"),
    ("NotRightTriangleBar;", "\u{29D0}\u{338}"),
    ("NotRightTriangleEqual;", "\u{22ED}"),
    ("NotSquareSubset;", "\u{228F}\u{338}"),
    ("NotSquareSubsetEqual;", "\u{22E2}"),
    ("NotSquareSuperset;", "\u{2290}\u{338}"),
    ("NotSquareSupersetEqual;", "\u{22E3}"),
    ("NotSubset;", "\u{2282}\u{20D2}"),
    ("NotSubsetEqual;", "\u{2288}"),
    ("NotSucceeds;", "\u{2281}"),
    ("NotSucceedsEqual;", "\u{2AB0}\u{338}"),
    ("NotSucceedsSlantEqual;", "\u{22E1}"),
    ("NotSucceedsTilde;", "\u{227F}\u{338}"),
    ("NotSuperset;", "\u{2283}\u{20D2}"),
    ("NotSupersetEqual;", "\u{2289}"),
    ("NotTilde;", "\u{2241}"),
    ("NotTildeEqual;", "\u{2244}"),
    ("NotTildeFullEqual;", "\u{2247}"),
    ("NotTildeTilde;", "\u{2249}"),
    ("NotVerticalBar;", "\u{2224}"),
    ("Nscr;", "\u{1D4A9}"),
    ("Ntilde", "\u{D1}"),
    ("Ntilde;", "\u{D1}"),
    ("Nu;", "\u{39D}"),
    ("OElig;", "\u{152}"),
    ("Oacute", "\u{D3}"),
    ("Oacute;", "\u{D3}"),
    ("Ocirc", "\u{D4}"),
    ("Ocirc;", "\u{D4}"),
    ("Ocy;", "\u{41E}"),
    ("Odblac;", "\u{150}"),
    ("Ofr;", "\u{1D512}"),
    ("Ograve", "\u{D2}"),
    ("Ograve;", "\u{D2}"),
    ("Omacr;", "\u{14C}"),
    ("Omega;", "\u{3A9}"),
    ("Omicron;", "\u{39F}"),
    ("Oopf;", "\u{1D546}"),
    ("OpenCurlyDoubleQuote;", "\u{201C}"),
    ("OpenCurlyQuote;", "\u{2018}"),
    ("Or;", "\u{2A54}"),
    ("Oscr;", "\u{1D4AA}"),
    ("Oslash", "\u{D8}"),
    ("Oslash;", "\u{D8}"),
    ("Otilde", "\u{D5}"),
    ("Otilde;", "\u{D5}"),
    ("Otimes;", "\u{2A37}"),
    ("Ouml", "\u{D6}"),
    ("Ouml;", "\u{D6}"),
    ("OverBar;", "\u{203E}"),
    ("OverBrace;", "\u{23DE}"),
    ("OverBracket;", "\u{23B4}"),
    ("OverParenthesis;", "\u{23DC}"),
    ("PartialD;", "\u{2202}"),
    ("Pcy;", "\u{41F}"),
    ("Pfr;", "\u{1D513}"),
    ("Phi;", "\u{3A6}"),
    ("Pi;", "\u{3A0}"),
    ("PlusMinus;", "\u{B1}"),
    ("Poincareplane;", "\u{210C}"),
    ("Popf;", "\u{2119}"),
    ("Pr;", "\u{2ABB}"),
    ("Precedes;", "\u{227A}"),
    ("PrecedesEqual;", "\u{2AAF}"),
    ("PrecedesSlantEqual;", "\u{227C}"),
    ("PrecedesTilde;", "\u{227E}"),
    ("Prime;", "\u{2033}"),
    ("Product;", "\u{220F}"),
    ("Proportion;", "\u{2237}"),
    ("Proportional;", "\u{221D}"),
    ("Pscr;", "\u{1D4AB}"),
    ("Psi;", "\u{3A8}"),
    ("QUOT", "\u{22}"),
    ("QUOT;", "\u{22}"),
    ("Qfr;", "\u{1D514}"),
    ("Qopf;", "\u{211A}"),
    ("Qscr;", "\u{1D4AC}"),
    ("RBarr;", "\u{2910}"),
    ("REG", "\u{AE}"),
    ("REG;", "\u{AE}"),
    ("Racute;", "\u{154}"),
    ("Rang;", "\u{27EB}"),
    ("Rarr;", "\u{21A0}"),
    ("Rarrtl;", "\u{2916}"),
    ("Rcaron;", "\u{158}"),
    ("Rcedil;", "\u{156}"),
    ("Rcy;", "\u{420}"),
    ("Re;", "\u{211C}"),
    ("ReverseElement;", "\u{220B}"),
    ("ReverseEquilibrium;", "\u{21CB}"),
    ("ReverseUpEquilibrium;", "\u{296F}"),
    ("Rfr;", "\u{211C}"),
    ("Rho;", "\u{3A1}"),
    ("RightAngleBracket;", "\u{27E9}"),
    ("RightArrow;", "\u{2192}"),
    ("RightArrowBar;", "\u{21E5}"),
    ("RightArrowLeftArrow;", "\u{21C4}"),
    ("RightCeiling;", "\u{2309}"),
    ("RightDoubleBracket;", "\u{27E7}"),
    ("RightDownTeeVector;", "\u{295D}"),
    ("RightDownVector;", "\u{21C2}"),
    ("RightDownVectorBar;", "\u{2955}"),
    ("RightFloor;", "\u{230B}"),
    ("RightTee;", "\u{22A2}"),
    ("RightTeeArrow;", "\u{21A6}"),
    ("RightTeeVector;", "\u{295B}"),
    ("RightTriangle;", "\u{22B3}"),
    ("RightTriangleBar;", "\u{29D0}"),
    ("RightTriangleEqual;", "\u{22B5}"),
    ("RightUpDownVector;", "\u{294F}"),
    ("RightUpTeeVector;", "\u{295C}"),
    ("RightUpVector;", "\u{21BE}"),
    ("RightUpVectorBar;", "\u{2954}"),
    ("RightVector;", "\u{21C0}"),
    ("RightVectorBar;", "\u{2953}"),
    ("Rightarrow;", "\u{21D2}"),
    ("Ropf;", "\u{211D}"),
    ("RoundImplies;", "\u{2970}"),
    ("Rrightarrow;", "\u{21DB}"),
    ("Rscr;", "\u{211B}"),
    ("Rsh;", "\u{21B1}"),
    ("RuleDelayed;", "\u{29F4}"),
    ("SHCHcy;", "\u{429}"),
    ("SHcy;", "\u{428}"),
    ("SOFTcy;", "\u{42C}"),
    ("Sacute;", "\u{15A}"),
    ("Sc;", "\u{2ABC}"),
    ("Scaron;", "\u{160}"),
    ("Scedil;", "\u{15E}"),
    ("Scirc;", "\u{15C}"),
    ("Scy;", "\u{421}"),
    ("Sfr;", "\u{1D516}"),
    ("ShortDownArrow;", "\u{2193}"),
    ("ShortLeftArrow;", "\u{2190}"),
    ("ShortRightArrow;", "\u{2192}"),
    ("ShortUpArrow;", "\u{2191}"),
    ("Sigma;", "\u{3A3}"),
    ("SmallCircle;", "\u{2218}"),
    ("Sopf;", "\u{1D54A}"),
    ("Sqrt;", "\u{221A}"),
    ("Square;", "\u{25A1}"),
    ("SquareIntersection;", "\u{2293}"),
    ("SquareSubset;", "\u{228F}"),
    ("SquareSubsetEqual;", "\u{2291}"),
    ("SquareSuperset;", "\u{2290}"),
    ("SquareSupersetEqual;", "\u{2292}"),
    ("SquareUnion;", "\u{2294}"),
    ("Sscr;", "\u{1D4AE}"),
    ("Star;", "\u{22C6}"),
    ("Sub;", "\u{22D0}"),
    ("Subset;", "\u{22D0}"),
    ("SubsetEqual;", "\u{2286}"),
    ("Succeeds;", "\u{227B}"),
    ("SucceedsEqual;", "\u{2AB0}"),
    ("SucceedsSlantEqual;", "\u{227D}"),
    ("SucceedsTilde;", "\u{227F}"),
    ("SuchThat;", "\u{220B}"),
    ("Sum;", "\u{2211}"),
    ("Sup;", "\u{22D1}"),
    ("Superset;", "\u{2283}"),
    ("SupersetEqual;", "\u{2287}"),
    ("Supset;", "\u{22D1}"),
    ("THORN", "\u{DE}"),
    ("THORN;", "\u{DE}"),
    ("TRADE;", "\u{2122}"),
    ("TSHcy;", "\u{40B}"),
    ("TScy;", "\u{426}"),
    ("Tab;", "\u{9}"),
    ("Tau;", "\u{3A4}"),
    ("Tcaron;", "\u{164}"),
    ("Tcedil;", "\u{162}"),
    ("Tcy;", "\u{422}"),
    ("Tfr;", "\u{1D517}"),
    ("Therefore;", "\u{2234}"),
    ("Theta;", "\u{398}"),
    ("ThickSpace;", "\u{205F}\u{200A}"),
    ("ThinSpace;", "\u{2009}"),
    ("Tilde;", "\u{223C}"),
    ("TildeEqual;", "\u{2243}"),
    ("TildeFullEqual;", "\u{2245}"),
    ("TildeTilde;", "\u{2248}"),
    ("Topf;", "\u{1D54B}"),
    ("TripleDot;", "\u{20DB}"),
    ("Tscr;", "\u{1D4AF}"),
    ("Tstrok;", "\u{166}"),
    ("Uacute", "\u{DA}"),
    ("Uacute;", "\u{DA}"),
    ("Uarr;", "\u{219F}"),
    ("Uarrocir;", "\u{2949}"),
    ("Ubrcy;", "\u{40E}"),
    ("Ubreve;", "\u{16C}"),
    ("Ucirc", "\u{DB}"),
    ("Ucirc;", "\u{DB}"),
    ("Ucy;", "\u{423}"),
    ("Udblac;", "\u{170}"),
    ("Ufr;", "\u{1D518}"),
    ("Ugrave", "\u{D9}"),
    ("Ugrave;", "\u{D9}"),
    ("Umacr;", "\u{16A}"),
    ("UnderBar;", "\u{5F}"),
    ("UnderBrace;", "\u{23DF}"),
    ("UnderBracket;", "\u{23B5}"),
    ("UnderParenthesis;", "\u{23DD}"),
    ("Union;", "\u{22C3}"),
    ("UnionPlus;", "\u{228E}"),
    ("Uogon;", "\u{172}"),
    ("Uopf;", "\u{1D54C}"),
    ("UpArrow;", "\u{2191}"),
    ("UpArrowBar;", "\u{2912}"),
    ("UpArrowDownArrow;", "\u{21C5}"),
    ("UpDownArrow;", "\u{2195}"),
    ("UpEquilibrium;", "\u{296E}"),
    ("UpTee;", "\u{22A5}"),
    ("UpTeeArrow;", "\u{21A5}"),
    ("Uparrow;", "\u{21D1}"),
    ("Updownarrow;", "\u{21D5}"),
    ("UpperLeftArrow;", "\u{2196}"),
    ("UpperRightArrow;", "\u{2197}"),
    ("Upsi;", "\u{3D2}"),
    ("Upsilon;", "\u{3A5}"),
    ("Uring;", "\u{16E}"),
    ("Uscr;", "\u{1D4B0}"),
    ("Utilde;", "\u{168}"),
    ("Uuml", "\u{DC}"),
    ("Uuml;", "\u{DC}"),
    ("VDash;", "\u{22AB}"),
    ("Vbar;", "\u{2AEB}"),
    ("Vcy;", "\u{412}"),
    ("Vdash;", "\u{22A9}"),
    ("Vdashl;", "\u{2AE6}"),
    ("Vee;", "\u{22C1}"),
    ("Verbar;", "\u{2016}"),
    ("Vert;", "\u{2016}"),
    ("VerticalBar;", "\u{2223}"),
    ("VerticalLine;", "\u{7C}"),
    ("VerticalSeparator;", "\u{2758}"),
    ("VerticalTilde;", "\u{2240}"),
    ("VeryThinSpace;", "\u{200A}"),
    ("Vfr;", "\u{1D519}"),
    ("Vopf;", "\u{1D54D}"),
    ("Vscr;", "\u{1D4B1}"),
    ("Vvdash;", "\u{22AA}"),
    ("Wcirc;", "\u{174}"),
    ("Wedge;", "\u{22C0}"),
    ("Wfr;", "\u{1D51A}"),
    ("Wopf;", "\u{1D54E}"),
    ("Wscr;", "\u{1D4B2}"),
    ("Xfr;", "\u{1D51B}"),
    ("Xi;", "\u{39E}"),
    ("Xopf;", "\u{1D54F}"),
    ("Xscr;", "\u{1D4B3}"),
    ("YAcy;", "\u{42F}"),
    ("YIcy;", "\u{407}"),
    ("YUcy;", "\u{42E}"),
    ("Yacute", "\u{DD}"),
    ("Yacute;", "\u{DD}"),
    ("Ycirc;", "\u{176}"),
    ("Ycy;", "\u{42B}"),
    ("Yfr;", "\u{1D51C}"),
    ("Yopf;", "\u{1D550}"),
    ("Yscr;", "\u{1D4B4}"),
    ("Yuml;", "\u{178}"),
    ("ZHcy;", "\u{416}"),
    ("Zacute;", "\u{179}"),
    ("Zcaron;", "\u{17D}"),
    ("Zcy;", "\u{417}"),
    ("Zdot;", "\u{17B}"),
    ("ZeroWidthSpace;", "\u{200B}"),
    ("Zeta;", "\u{396}"),
    ("Zfr;", "\u{2128}"),
    ("Zopf;", "\u{2124}"),
    ("Zscr;", "\u{1D4B5}"),
    ("aacute", "\u{E1}"),
    ("aacute;", "\u{E1}"),
    ("abreve;", "\u{103}"),
    ("ac;", "\u{223E}"),
    ("acE;", "\u{223E}\u{333}"),
    ("acd;", "\u{223F}"),
    ("acirc", "\u{E2}"),
    ("acirc;", "\u{E2}"),
    ("acute", "\u{B4}"),
    ("acute;", "\u{B4}"),
    ("acy;", "\u{430}"),
    ("aelig", "\u{E6}"),
    ("aelig;", "\u{E6}"),
    ("af;", "\u{2061}"),
    ("afr;", "\u{1D51E}"),
    ("agrave", "\u{E0}"),
    ("agrave;", "\u{E0}"),
    ("alefsym;", "\u{2135}"),
    ("aleph;", "\u{2135}"),
    ("alpha;", "\u{3B1}"),
    ("amacr;", "\u{101}"),
    ("amalg;", "\u{2A3F}"),
    ("amp", "\u{26}"),
    ("amp;", "\u{26}"),
    ("and;", "\u{2227}"),
    ("andand;", "\u{2A55}"),
    ("andd;", "\u{2A5C}"),
    ("andslope;", "\u{2A58}"),
    ("andv;", "\u{2A5A}"),
    ("ang;", "\u{2220}"),
    ("ange;", "\u{29A4}"),
    ("angle;", "\u{2220}"),
    ("angmsd;", "\u{2221}"),
    ("angmsdaa;", "\u{29A8}"),
    ("angmsdab;", "\u{29A9}"),
    ("angmsdac;", "\u{29AA}"),
    ("angmsdad;", "\u{29AB}"),
    ("angmsdae;", "\u{29AC}"),
    ("angmsdaf;", "\u{29AD}"),
    ("angmsdag;", "\u{29AE}"),
    ("angmsdah;", "\u{29AF}"),
    ("angrt;", "\u{221F}"),
    ("angrtvb;", "\u{22BE}"),
    ("angrtvbd;", "\u{299D}"),
    ("angsph;", "\u{2222}"),
    ("angst;", "\u{C5}"),
    ("angzarr;", "\u{237C}"),
    ("aogon;", "\u{105}"),
    ("aopf;", "\u{1D552}"),
    ("ap;", "\u{2248}"),
    ("apE;", "\u{2A70}"),
    ("apacir;", "\u{2A6F}"),
    ("ape;", "\u{224A}"),
    ("apid;", "\u{224B}"),
    ("apos;", "\u{27}"),
    ("approx;", "\u{2248}"),
    ("approxeq;", "\u{224A}"),
    ("aring", "\u{E5}"),
    ("aring;", "\u{E5}"),
    ("ascr;", "\u{1D4B6}"),
    ("ast;", "\u{2A}"),
    ("asymp;", "\u{2248}"),
    ("asympeq;", "\u{224D}"),
    ("atilde", "\u{E3}"),
    ("atilde;", "\u{E3}"),
    ("auml", "\u{E4}"),
    ("auml;", "\u{E4}"),
    ("awconint;", "\u{2233}"),
    ("awint;", "\u{2A11}"),
    ("bNot;", "\u{2AED}"),
    ("backcong;", "\u{224C}"),
    ("backepsilon;", "\u{3F6}"),
    ("backprime;", "\u{2035}"),
    ("backsim;", "\u{223D}"),
    ("backsimeq;", "\u{22CD}"),
    ("barvee;", "\u{22BD}"),
    ("barwed;", "\u{2305}"),
    ("barwedge;", "\u{2305}"),
    ("bbrk;", "\u{23B5}"),
    ("bbrktbrk;", "\u{23B6}"),
    ("bcong;", "\u{224C}"),
    ("bcy;", "\u{431}"),
    ("bdquo;", "\u{201E}"),
    ("becaus;", "\u{2235}"),
    ("because;", "\u{2235}"),
    ("bemptyv;", "\u{29B0}"),
    ("bepsi;", "\u{3F6}"),
    ("bernou;", "\u{212C}"),
    ("beta;", "\u{3B2}"),
    ("beth;", "\u{2136}"),
    ("between;", "\u{226C}"),
    ("bfr;", "\u{1D51F}"),
    ("bigcap;", "\u{22C2}"),
    ("bigcirc;", "\u{25EF}"),
    ("bigcup;", "\u{22C3}"),
    ("bigodot;", "\u{2A00}"),
    ("bigoplus;", "\u{2A01}"),
    ("bigotimes;", "\u{2A02}"),
    ("bigsqcup;", "\u{2A06}"),
    ("bigstar;", "\u{2605}"),
    ("bigtriangledown;", "\u{25BD}"),
    ("bigtriangleup;", "\u{25B3}"),
    ("biguplus;", "\u{2A04}"),
    ("bigvee;", "\u{22C1}"),
    ("bigwedge;", "\u{22C0}"),
    ("bkarow;", "\u{290D}"),
    ("blacklozenge;", "\u{29EB}"),
    ("blacksquare;", "\u{25AA}"),
    ("blacktriangle;", "\u{25B4}"),
    ("blacktriangledown;", "\u{25BE}"),
    ("blacktriangleleft;", "\u{25C2}"),
    ("blacktriangleright;", "\u{25B8}"),
    ("blank;", "\u{2423}"),
    ("blk12;", "\u{2592}"),
    ("blk14;", "\u{2591}"),
    ("blk34;", "\u{2593}"),
    ("block;", "\u{2588}"),
    ("bne;", "\u{3D}\u{20E5}"),
    ("bnequiv;", "\u{2261}\u{20E5}"),
    ("bnot;", "\u{2310}"),
    ("bopf;", "\u{1D553}"),
    ("bot;", "\u{22A5}"),
    ("bottom;", "\u{22A5}"),
    ("bowtie;", "\u{22C8}"),
    ("boxDL;", "\u{2557}"),
    ("boxDR;", "\u{2554}"),
    ("boxDl;", "\u{2556}"),
    ("boxDr;", "\u{2553}"),
    ("boxH;", "\u{2550}"),
    ("boxHD;", "\u{2566}"),
    ("boxHU;", "\u{2569}"),
    ("boxHd;", "\u{2564}"),
    ("boxHu;", "\u{2567}"),
    ("boxUL;", "\u{255D}"),
    ("boxUR;", "\u{255A}"),
    ("boxUl;", "\u{255C}"),
    ("boxUr;", "\u{2559}"),
    ("boxV;", "\u{2551}"),
    ("boxVH;", "\u{256C}"),
    ("boxVL;", "\u{2563}"),
    ("boxVR;", "\u{2560}"),
    ("boxVh;", "\u{256B}"),
    ("boxVl;", "\u{2562}"),
    ("boxVr;", "\u{255F}"),
    ("boxbox;", "\u{29C9}"),
    ("boxdL;", "\u{2555}"),
    ("boxdR;", "\u{2552}"),
    ("boxdl;", "\u{2510}"),
    ("boxdr;", "\u{250C}"),
    ("boxh;", "\u{2500}"),
    ("boxhD;", "\u{2565}"),
    ("boxhU;", "\u{2568}"),
    ("boxhd;", "\u{252C}"),
    ("boxhu;", "\u{2534}"),
    ("boxminus;", "\u{229F}"),
    ("boxplus;", "\u{229E}"),
    ("boxtimes;", "\u{22A0}"),
    ("boxuL;", "\u{255B}"),
    ("boxuR;", "\u{2558}"),
    ("boxul;", "\u{2518}"),
    ("boxur;", "\u{2514}"),
    ("boxv;", "\u{2502}"),
    ("boxvH;", "\u{256A}"),
    ("boxvL;", "\u{2561}"),
    ("boxvR;", "\u{255E}"),
    ("boxvh;", "\u{253C}"),
    ("boxvl;", "\u{2524}"),
    ("boxvr;", "\u{251C}"),
    ("bprime;", "\u{2035}"),
    ("breve;", "\u{2D8}"),
    ("brvbar", "\u{A6}"),
    ("brvbar;", "\u{A6}"),
    ("bscr;", "\u{1D4B7}"),
    ("bsemi;", "\u{204F}"),
    ("bsim;", "\u{223D}"),
    ("bsime;", "\u{22CD}"),
    ("bsol;", "\u{5C}"),
    ("bsolb;", "\u{29C5}"),
    ("bsolhsub;", "\u{27C8}"),
    ("bull;", "\u{2022}"),
    ("bullet;", "\u{2022}"),
    ("bump;", "\u{224E}"),
    ("bumpE;", "\u{2AAE}"),
    ("bumpe;", "\u{224F}"),
    ("bumpeq;", "\u{224F}"),
    ("cacute;", "\u{107}"),
    ("cap;", "\u{2229}"),
    ("capand;", "\u{2A44}"),
    ("capbrcup;", "\u{2A49}"),
    ("capcap;", "\u{2A4B}"),
    ("capcup;", "\u{2A47}"),
    ("capdot;", "\u{2A40}"),
    ("caps;", "\u{2229}\u{FE00}"),
    ("caret;", "\u{2041}"),
    ("caron;", "\u{2C7}"),
    ("ccaps;", "\u{2A4D}"),
    ("ccaron;", "\u{10D}"),
    ("ccedil", "\u{E7}"),
    ("ccedil;", "\u{E7}"),
    ("ccirc;", "\u{109}"),
    ("ccups;", "\u{2A4C}"),
    ("ccupssm;", "\u{2A50}"),
    ("cdot;", "\u{10B}"),
    ("cedil", "\u{B8}"),
    ("cedil;", "\u{B8}"),
    ("cemptyv;", "\u{29B2}"),
    ("cent", "\u{A2}"),
    ("cent;", "\u{A2}"),
    ("centerdot;", "\u{B7}"),
    ("cfr;", "\u{1D520}"),
    ("chcy;", "\u{447}"),
    ("check;", "\u{2713}"),
    ("checkmark;", "\u{2713}"),
    ("chi;", "\u{3C7}"),
    ("cir;", "\u{25CB}"),
    ("cirE;", "\u{29C3}"),
    ("circ;", "\u{2C6}"),
    ("circeq;", "\u{2257}"),
    ("circlearrowleft;", "\u{21BA}"),
    ("circlearrowright;", "\u{21BB}"),
    ("circledR;", "\u{AE}"),
    ("circledS;", "\u{24C8}"),
    ("circledast;", "\u{229B}"),
    ("circledcirc;", "\u{229A}"),
    ("circleddash;", "\u{229D}"),
    ("cire;", "\u{2257}"),
    ("cirfnint;", "\u{2A10}"),
    ("cirmid;", "\u{2AEF}"),
    ("cirscir;", "\u{29C2}"),
    ("clubs;", "\u{2663}"),
    ("clubsuit;", "\u{2663}"),
    ("colon;", "\u{3A}"),
    ("colone;", "\u{2254}"),
    ("coloneq;", "\u{2254}"),
    ("comma;", "\u{2C}"),
    ("commat;", "\u{40}"),
    ("comp;", "\u{2201}"),
    ("compfn;", "\u{2218}"),
    ("complement;", "\u{2201}"),
    ("complexes;", "\u{2102}"),
    ("cong;", "\u{2245}"),
    ("congdot;", "\u{2A6D}"),
    ("conint;", "\u{222E}"),
    ("copf;", "\u{1D554}"),
    ("coprod;", "\u{2210}"),
    ("copy", "\u{A9}"),
    ("copy;", "\u{A9}"),
    ("copysr;", "\u{2117}"),
    ("crarr;", "\u{21B5}"),
    ("cross;", "\u{2717}"),
    ("cscr;", "\u{1D4B8}"),
    ("csub;", "\u{2ACF}"),
    ("csube;", "\u{2AD1}"),
    ("csup;", "\u{2AD0}"),
    ("csupe;", "\u{2AD2}"),
    ("ctdot;", "\u{22EF}"),
    ("cudarrl;", "\u{2938}"),
    ("cudarrr;", "\u{2935}"),
    ("cuepr;", "\u{22DE}"),
    ("cuesc;", "\u{22DF}"),
    ("cularr;", "\u{21B6}"),
    ("cularrp;", "\u{293D}"),
    ("cup;", "\u{222A}"),
    ("cupbrcap;", "\u{2A48}"),
    ("cupcap;", "\u{2A46}"),
    ("cupcup;", "\u{2A4A}"),
    ("cupdot;", "\u{228D}"),
    ("cupor;", "\u{2A45}"),
    ("cups;", "\u{222A}\u{FE00}"),
    ("curarr;", "\u{21B7}"),
    ("curarrm;", "\u{293C}"),
    ("curlyeqprec;", "\u{22DE}"),
    ("curlyeqsucc;", "\u{22DF}"),
    ("curlyvee;", "\u{22CE}"),
    ("curlywedge;", "\u{22CF}"),
    ("curren", "\u{A4}"),
    ("curren;", "\u{A4}"),
    ("curvearrowleft;", "\u{21B6}"),
    ("curvearrowright;", "\u{21B7}"),
    ("cuvee;", "\u{22CE}"),
    ("cuwed;", "\u{22CF}"),
    ("cwconint;", "\u{2232}"),
    ("cwint;", "\u{2231}"),
    ("cylcty;", "\u{232D}"),
    ("dArr;", "\u{21D3}"),
    ("dHar;", "\u{2965}"),
    ("dagger;", "\u{2020}"),
    ("daleth;", "\u{2138}"),
    ("darr;", "\u{2193}"),
    ("dash;", "\u{2010}"),
    ("dashv;", "\u{22A3}"),
    ("dbkarow;", "\u{290F}"),
    ("dblac;", "\u{2DD}"),
    ("dcaron;", "\u{10F}"),
    ("dcy;", "\u{434}"),
    ("dd;", "\u{2146}"),
    ("ddagger;", "\u{2021}"),
    ("ddarr;", "\u{21CA}"),
    ("ddotseq;", "\u{2A77}"),
    ("deg", "\u{B0}"),
    ("deg;", "\u{B0}"),
    ("delta;", "\u{3B4}"),
    ("demptyv;", "\u{29B1}"),
    ("dfisht;", "\u{297F}"),
    ("dfr;", "\u{1D521}"),
    ("dharl;", "\u{21C3}"),
    ("dharr;", "\u{21C2}"),
    ("diam;", "\u{22C4}"),
    ("diamond;", "\u{22C4}"),
    ("diamondsuit;", "\u{2666}"),
    ("diams;", "\u{2666}"),
    ("die;", "\u{A8}"),
    ("digamma;", "\u{3DD}"),
    ("disin;", "\u{22F2}"),
    ("div;", "\u{F7}"),
    ("divide", "\u{F7}"),
    ("divide;", "\u{F7}"),
    ("divideontimes;", "\u{22C7}"),
    ("divonx;", "\u{22C7}"),
    ("djcy;", "\u{452}"),
    ("dlcorn;", "\u{231E}"),
    ("dlcrop;", "\u{230D}"),
    ("dollar;", "\u{24}"),
    ("dopf;", "\u{1D555}"),
    ("dot;", "\u{2D9}"),
    ("doteq;", "\u{2250}"),
    ("doteqdot;", "\u{2251}"),
    ("dotminus;", "\u{2238}"),
    ("dotplus;", "\u{2214}"),
    ("dotsquare;", "\u{22A1}"),
    ("doublebarwedge;", "\u{2306}"),
    ("downarrow;", "\u{2193}"),
    ("downdownarrows;", "\u{21CA}"),
    ("downharpoonleft;", "\u{21C3}"),
    ("downharpoonright;", "\u{21C2}"),
    ("drbkarow;", "\u{2910}"),
    ("drcorn;", "\u{231F}"),
    ("drcrop;", "\u{230C}"),
    ("dscr;", "\u{1D4B9}"),
    ("dscy;", "\u{455}"),
    ("dsol;", "\u{29F6}"),
    ("dstrok;", "\u{111}"),
    ("dtdot;", "\u{22F1}"),
    ("dtri;", "\u{25BF}"),
    ("dtrif;", "\u{25BE}"),
    ("duarr;", "\u{21F5}"),
    ("duhar;", "\u{296F}"),
    ("dwangle;", "\u{29A6}"),
    ("dzcy;", "\u{45F}"),
    ("dzigrarr;", "\u{27FF}"),
    ("eDDot;", "\u{2A77}"),
    ("eDot;", "\u{2251}"),
    ("eacute", "\u{E9}"),
    ("eacute;", "\u{E9}"),
    ("easter;", "\u{2A6E}"),
    ("ecaron;", "\u{11B}"),
    ("ecir;", "\u{2256}"),
    ("ecirc", "\u{EA}"),
    ("ecirc;", "\u{EA}"),
    ("ecolon;", "\u{2255}"),
    ("ecy;", "\u{44D}"),
    ("edot;", "\u{117}"),
    ("ee;", "\u{2147}"),
    ("efDot;", "\u{2252}"),
    ("efr;", "\u{1D522}"),
    ("eg;", "\u{2A9A}"),
    ("egrave", "\u{E8}"),
    ("egrave;", "\u{E8}"),
    ("egs;", "\u{2A96}"),
    ("egsdot;", "\u{2A98}"),
    ("el;", "\u{2A99}"),
    ("elinters;", "\u{23E7}"),
    ("ell;", "\u{2113}"),
    ("els;", "\u{2A95}"),
    ("elsdot;", "\u{2A97}"),
    ("emacr;", "\u{113}"),
    ("empty;", "\u{2205}"),
    ("emptyset;", "\u{2205}"),
    ("emptyv;", "\u{2205}"),
    ("emsp13;", "\u{2004}"),
    ("emsp14;", "\u{2005}"),
    ("emsp;", "\u{2003}"),
    ("eng;", "\u{14B}"),
    ("ensp;", "\u{2002}"),
    ("eogon;", "\u{119}"),
    ("eopf;", "\u{1D556}"),
    ("epar;", "\u{22D5}"),
    ("eparsl;", "\u{29E3}"),
    ("eplus;", "\u{2A71}"),
    ("epsi;", "\u{3B5}"),
    ("epsilon;", "\u{3B5}"),
    ("epsiv;", "\u{3F5}"),
    ("eqcirc;", "\u{2256}"),
    ("eqcolon;", "\u{2255}"),
    ("eqsim;", "\u{2242}"),
    ("eqslantgtr;", "\u{2A96}"),
    ("eqslantless;", "\u{2A95}"),
    ("equals;", "\u{3D}"),
    ("equest;", "\u{225F}"),
    ("equiv;", "\u{2261}"),
    ("equivDD;", "\u{2A78}"),
    ("eqvparsl;", "\u{29E5}"),
    ("erDot;", "\u{2253}"),
    ("erarr;", "\u{2971}"),
    ("escr;", "\u{212F}"),
    ("esdot;", "\u{2250}"),
    ("esim;", "\u{2242}"),
    ("eta;", "\u{3B7}"),
    ("eth", "\u{F0}"),
    ("eth;", "\u{F0}"),
    ("euml", "\u{EB}"),
    ("euml;", "\u{EB}"),
    ("euro;", "\u{20AC}"),
    ("excl;", "\u{21}"),
    ("exist;", "\u{2203}"),
    ("expectation;", "\u{2130}"),
    ("exponentiale;", "\u{2147}"),
    ("fallingdotseq;", "\u{2252}"),
    ("fcy;", "\u{444}"),
    ("female;", "\u{2640}"),
    ("ffilig;", "\u{FB03}"),
    ("fflig;", "\u{FB00}"),
    ("ffllig;", "\u{FB04}"),
    ("ffr;", "\u{1D523}"),
    ("filig;", "\u{FB01}"),
    ("fjlig;", "\u{66}\u{6A}"),
    ("flat;", "\u{266D}"),
    ("fllig;", "\u{FB02}"),
    ("fltns;", "\u{25B1}"),
    ("fnof;", "\u{192}"),
    ("fopf;", "\u{1D557}"),
    ("forall;", "\u{2200}"),
    ("fork;", "\u{22D4}"),
    ("forkv;", "\u{2AD9}"),
    ("fpartint;", "\u{2A0D}"),
    ("frac12", "\u{BD}"),
    ("frac12;", "\u{BD}"),
    ("frac13;", "\u{2153}"),
    ("frac14", "\u{BC}"),
    ("frac14;", "\u{BC}"),
    ("frac15;", "\u{2155}"),
    ("frac16;", "\u{2159}"),
    ("frac18;", "\u{215B}"),
    ("frac23;", "\u{2154}"),
    ("frac25;", "\u{2156}"),
    ("frac34", "\u{BE}"),
    ("frac34;", "\u{BE}"),
    ("frac35;", "\u{2157}"),
    ("frac38;", "\u{215C}"),
    ("frac45;", "\u{2158}"),
    ("frac56;", "\u{215A}"),
    ("frac58;", "\u{215D}"),
    ("frac78;", "\u{215E}"),
    ("frasl;", "\u{2044}"),
    ("frown;", "\u{2322}"),
    ("fscr;", "\u{1D4BB}"),
    ("gE;", "\u{2267}"),
    ("gEl;", "\u{2A8C}"),
    ("gacute;", "\u{1F5}"),
    ("gamma;", "\u{3B3}"),
    ("gammad;", "\u{3DD}"),
    ("gap;", "\u{2A86}"),
    ("gbreve;", "\u{11F}"),
    ("gcirc;", "\u{11D}"),
    ("gcy;", "\u{433}"),
    ("gdot;", "\u{121}"),
    ("ge;", "\u{2265}"),
    ("gel;", "\u{22DB}"),
    ("geq;", "\u{2265}"),
    ("geqq;", "\u{2267}"),
    ("geqslant;", "\u{2A7E}"),
    ("ges;", "\u{2A7E}"),
    ("gescc;", "\u{2AA9}"),
    ("gesdot;", "\u{2A80}"),
    ("gesdoto;", "\u{2A82}"),
    ("gesdotol;", "\u{2A84}"),
    ("gesl;", "\u{22DB}\u{FE00}"),
    ("gesles;", "\u{2A94}"),
    ("gfr;", "\u{1D524}"),
    ("gg;", "\u{226B}"),
    ("ggg;", "\u{22D9}"),
    ("gimel;", "\u{2137}"),
    ("gjcy;", "\u{453}"),
    ("gl;", "\u{2277}"),
    ("glE;", "\u{2A92}"),
    ("gla;", "\u{2AA5}"),
    ("glj;", "\u{2AA4}"),
    ("gnE;", "\u{2269}"),
    ("gnap;", "\u{2A8A}"),
    ("gnapprox;", "\u{2A8A}"),
    ("gne;", "\u{2A88}"),
    ("gneq;", "\u{2A88}"),
    ("gneqq;", "\u{2269}"),
    ("gnsim;", "\u{22E7}"),
    ("gopf;", "\u{1D558}"),
    ("grave;", "\u{60}"),
    ("gscr;", "\u{210A}"),
    ("gsim;", "\u{2273}"),
    ("gsime;", "\u{2A8E}"),
    ("gsiml;", "\u{2A90}"),
    ("gt", "\u{3E}"),
    ("gt;", "\u{3E}"),
    ("gtcc;", "\u{2AA7}"),
    ("gtcir;", "\u{2A7A}"),
    ("gtdot;", "\u{22D7}"),
    ("gtlPar;", "\u{2995}"),
    ("gtquest;", "\u{2A7C}"),
    ("gtrapprox;", "\u{2A86}"),
    ("gtrarr;", "\u{2978}"),
    ("gtrdot;", "\u{22D7}"),
    ("gtreqless;", "\u{22DB}"),
    ("gtreqqless;", "\u{2A8C}"),
    ("gtrless;", "\u{2277}"),
    ("gtrsim;", "\u{2273}"),
    ("gvertneqq;", "\u{2269}\u{FE00}"),
    ("gvnE;", "\u{2269}\u{FE00}"),
    ("hArr;", "\u{21D4}"),
    ("hairsp;", "\u{200A}"),
    ("half;", "\u{BD}"),
    ("hamilt;", "\u{210B}"),
    ("hardcy;", "\u{44A}"),
    ("harr;", "\u{2194}"),
    ("harrcir;", "\u{2948}"),
    ("harrw;", "\u{21AD}"),
    ("hbar;", "\u{210F}"),
    ("hcirc;", "\u{125}"),
    ("hearts;", "\u{2665}"),
    ("heartsuit;", "\u{2665}"),
    ("hellip;", "\u{2026}"),
    ("hercon;", "\u{22B9}"),
    ("hfr;", "\u{1D525}"),
    ("hksearow;", "\u{2925}"),
    ("hkswarow;", "\u{2926}"),
    ("hoarr;", "\u{21FF}"),
    ("homtht;", "\u{223B}"),
    ("hookleftarrow;", "\u{21A9}"),
    ("hookrightarrow;", "\u{21AA}"),
    ("hopf;", "\u{1D559}"),
    ("horbar;", "\u{2015}"),
    ("hscr;", "\u{1D4BD}"),
    ("hslash;", "\u{210F}"),
    ("hstrok;", "\u{127}"),
    ("hybull;", "\u{2043}"),
    ("hyphen;", "\u{2010}"),
    ("iacute", "\u{ED}"),
    ("iacute;", "\u{ED}"),
    ("ic;", "\u{2063}"),
    ("icirc", "\u{EE}"),
    ("icirc;", "\u{EE}"),
    ("icy;", "\u{438}"),
    ("iecy;", "\u{435}"),
    ("iexcl", "\u{A1}"),
    ("iexcl;", "\u{A1}"),
    ("iff;", "\u{21D4}"),
    ("ifr;", "\u{1D526}"),
    ("igrave", "\u{EC}"),
    ("igrave;", "\u{EC}"),
    ("ii;", "\u{2148}"),
    ("iiiint;", "\u{2A0C}"),
    ("iiint;", "\u{222D}"),
    ("iinfin;", "\u{29DC}"),
    ("iiota;", "\u{2129}"),
    ("ijlig;", "\u{133}"),
    ("imacr;", "\u{12B}"),
    ("image;", "\u{2111}"),
    ("imagline;", "\u{2110}"),
    ("imagpart;", "\u{2111}"),
    ("imath;", "\u{131}"),
    ("imof;", "\u{22B7}"),
    ("imped;", "\u{1B5}"),
    ("in;", "\u{2208}"),
    ("incare;", "\u{2105}"),
    ("infin;", "\u{221E}"),
    ("infintie;", "\u{29DD}"),
    ("inodot;", "\u{131}"),
    ("int;", "\u{222B}"),
    ("intcal;", "\u{22BA}"),
    ("integers;", "\u{2124}"),
    ("intercal;", "\u{22BA}"),
    ("intlarhk;", "\u{2A17}"),
    ("intprod;", "\u{2A3C}"),
    ("iocy;", "\u{451}"),
    ("iogon;", "\u{12F}"),
    ("iopf;", "\u{1D55A}"),
    ("iota;", "\u{3B9}"),
    ("iprod;", "\u{2A3C}"),
    ("iquest", "\u{BF}"),
    ("iquest;", "\u{BF}"),
    ("iscr;", "\u{1D4BE}"),
    ("isin;", "\u{2208}"),
    ("isinE;", "\u{22F9}"),
    ("isindot;", "\u{22F5}"),
    ("isins;", "\u{22F4}"),
    ("isinsv;", "\u{22F3}"),
    ("isinv;", "\u{2208}"),
    ("it;", "\u{2062}"),
    ("itilde;", "\u{129}"),
    ("iukcy;", "\u{456}"),
    ("iuml", "\u{EF}"),
    ("iuml;", "\u{EF}"),
    ("jcirc;", "\u{135}"),
    ("jcy;", "\u{439}"),
    ("jfr;", "\u{1D527}"),
    ("jmath;", "\u{237}"),
    ("jopf;", "\u{1D55B}"),
    ("jscr;", "\u{1D4BF}"),
    ("jsercy;", "\u{458}"),
    ("jukcy;", "\u{454}"),
    ("kappa;", "\u{3BA}"),
    ("kappav;", "\u{3F0}"),
    ("kcedil;", "\u{137}"),
    ("kcy;", "\u{43A}"),
    ("kfr;", "\u{1D528}"),
    ("kgreen;", "\u{138}"),
    ("khcy;", "\u{445}"),
    ("kjcy;", "\u{45C}"),
    ("kopf;", "\u{1D55C}"),
    ("kscr;", "\u{1D4C0}"),
    ("lAarr;", "\u{21DA}"),
    ("lArr;", "\u{21D0}"),
    ("lAtail;", "\u{291B}"),
    ("lBarr;", "\u{290E}"),
    ("lE;", "\u{2266}"),
    ("lEg;", "\u{2A8B}"),
    ("lHar;", "\u{2962}"),
    ("lacute;", "\u{13A}"),
    ("laemptyv;", "\u{29B4}"),
    ("lagran;", "\u{2112}"),
    ("lambda;", "\u{3BB}"),
    ("lang;", "\u{27E8}"),
    ("langd;", "\u{2991}"),
    ("langle;", "\u{27E8}"),
    ("lap;", "\u{2A85}"),
    ("laquo", "\u{AB}"),
    ("laquo;", "\u{AB}"),
    ("larr;", "\u{2190}"),
    ("larrb;", "\u{21E4}"),
    ("larrbfs;", "\u{291F}"),
    ("larrfs;", "\u{291D}"),
    ("larrhk;", "\u{21A9}"),
    ("larrlp;", "\u{21AB}"),
    ("larrpl;", "\u{2939}"),
    ("larrsim;", "\u{2973}"),
    ("larrtl;", "\u{21A2}"),
    ("lat;", "\u{2AAB}"),
    ("latail;", "\u{2919}"),
    ("late;", "\u{2AAD}"),
    ("lates;", "\u{2AAD}\u{FE00}"),
    ("lbarr;", "\u{290C}"),
    ("lbbrk;", "\u{2772}"),
    ("lbrace;", "\u{7B}"),
    ("lbrack;", "\u{5B}"),
    ("lbrke;", "\u{298B}"),
    ("lbrksld;", "\u{298F}"),
    ("lbrkslu;", "\u{298D}"),
    ("lcaron;", "\u{13E}"),
    ("lcedil;", "\u{13C}"),
    ("lceil;", "\u{2308}"),
    ("lcub;", "\u{7B}"),
    ("lcy;", "\u{43B}"),
    ("ldca;", "\u{2936}"),
    ("ldquo;", "\u{201C}"),
    ("ldquor;", "\u{201E}"),
    ("ldrdhar;", "\u{2967}"),
    ("ldrushar;", "\u{294B}"),
    ("ldsh;", "\u{21B2}"),
    ("le;", "\u{2264}"),
    ("leftarrow;", "\u{2190}"),
    ("leftarrowtail;", "\u{21A2}"),
    ("leftharpoondown;", "\u{21BD}"),
    ("leftharpoonup;", "\u{21BC}"),
    ("leftleftarrows;", "\u{21C7}"),
    ("leftrightarrow;", "\u{2194}"),
    ("leftrightarrows;", "\u{21C6}"),
    ("leftrightharpoons;", "\u{21CB}"),
    ("leftrightsquigarrow;", "\u{21AD}"),
    ("leftthreetimes;", "\u{22CB}"),
    ("leg;", "\u{22DA}"),
    ("leq;", "\u{2264}"),
    ("leqq;", "\u{2266}"),
    ("leqslant;", "\u{2A7D}"),
    ("les;", "\u{2A7D}"),
    ("lescc;", "\u{2AA8}"),
    ("lesdot;", "\u{2A7F}"),
    ("lesdoto;", "\u{2A81}"),
    ("lesdotor;", "\u{2A83}"),
    ("lesg;", "\u{22DA}\u{FE00}"),
    ("lesges;", "\u{2A93}"),
    ("lessapprox;", "\u{2A85}"),
    ("lessdot;", "\u{22D6}"),
    ("lesseqgtr;", "\u{22DA}"),
    ("lesseqqgtr;", "\u{2A8B}"),
    ("lessgtr;", "\u{2276}"),
    ("lesssim;", "\u{2272}"),
    ("lfisht;", "\u{297C}"),
    ("lfloor;", "\u{230A}"),
    ("lfr;", "\u{1D529}"),
    ("lg;", "\u{2276}"),
    ("lgE;", "\u{2A91}"),
    ("lhard;", "\u{21BD}"),
    ("lharu;", "\u{21BC}"),
    ("lharul;", "\u{296A}"),
    ("lhblk;", "\u{2584}"),
    ("ljcy;", "\u{459}"),
    ("ll;", "\u{226A}"),
    ("llarr;", "\u{21C7}"),
    ("llcorner;", "\u{231E}"),
    ("llhard;", "\u{296B}"),
    ("lltri;", "\u{25FA}"),
    ("lmidot;", "\u{140}"),
    ("lmoust;", "\u{23B0}"),
    ("lmoustache;", "\u{23B0}"),
    ("lnE;", "\u{2268}"),
    ("lnap;", "\u{2A89}"),
    ("lnapprox;", "\u{2A89}"),
    ("lne;", "\u{2A87}"),
    ("lneq;", "\u{2A87}"),
    ("lneqq;", "\u{2268}"),
    ("lnsim;", "\u{22E6}"),
    ("loang;", "\u{27EC}"),
    ("loarr;", "\u{21FD}"),
    ("lobrk;", "\u{27E6}"),
    ("longleftarrow;", "\u{27F5}"),
    ("longleftrightarrow;", "\u{27F7}"),
    ("longmapsto;", "\u{27FC}"),
    ("longrightarrow;", "\u{27F6}"),
    ("looparrowleft;", "\u{21AB}"),
    ("looparrowright;", "\u{21AC}"),
    ("lopar;", "\u{2985}"),
    ("lopf;", "\u{1D55D}"),
    ("loplus;", "\u{2A2D}"),
    ("lotimes;", "\u{2A34}"),
    ("lowast;", "\u{2217}"),
    ("lowbar;", "\u{5F}"),
    ("loz;", "\u{25CA}"),
    ("lozenge;", "\u{25CA}"),
    ("lozf;", "\u{29EB}"),
    ("lpar;", "\u{28}"),
    ("lparlt;", "\u{2993}"),
    ("lrarr;", "\u{21C6}"),
    ("lrcorner;", "\u{231F}"),
    ("lrhar;", "\u{21CB}"),
    ("lrhard;", "\u{296D}"),
    ("lrm;", "\u{200E}"),
    ("lrtri;", "\u{22BF}"),
    ("lsaquo;", "\u{2039}"),
    ("lscr;", "\u{1D4C1}"),
    ("lsh;", "\u{21B0}"),
    ("lsim;", "\u{2272}"),
    ("lsime;", "\u{2A8D}"),
    ("lsimg;", "\u{2A8F}"),
    ("lsqb;", "\u{5B}"),
    ("lsquo;", "\u{2018}"),
    ("lsquor;", "\u{201A}"),
    ("lstrok;", "\u{142}"),
    ("lt", "\u{3C}"),
    ("lt;", "\u{3C}"),
    ("ltcc;", "\u{2AA6}"),
    ("ltcir;", "\u{2A79}"),
    ("ltdot;", "\u{22D6}"),
    ("lthree;", "\u{22CB}"),
    ("ltimes;", "\u{22C9}"),
    ("ltlarr;", "\u{2976}"),
    ("ltquest;", "\u{2A7B}"),
    ("ltrPar;", "\u{2996}"),
    ("ltri;", "\u{25C3}"),
    ("ltrie;", "\u{22B4}"),
    ("ltrif;", "\u{25C2}"),
    ("lurdshar;", "\u{294A}"),
    ("luruhar;", "\u{2966}"),
    ("lvertneqq;", "\u{2268}\u{FE00}"),
    ("lvnE;", "\u{2268}\u{FE00}"),
    ("mDDot;", "\u{223A}"),
    ("macr", "\u{AF}"),
    ("macr;", "\u{AF}"),
    ("male;", "\u{2642}"),
    ("malt;", "\u{2720}"),
    ("maltese;", "\u{2720}"),
    ("map;", "\u{21A6}"),
    ("mapsto;", "\u{21A6}"),
    ("mapstodown;", "\u{21A7}"),
    ("mapstoleft;", "\u{21A4}"),
    ("mapstoup;", "\u{21A5}"),
    ("marker;", "\u{25AE}"),
    ("mcomma;", "\u{2A29}"),
    ("mcy;", "\u{43C}"),
    ("mdash;", "\u{2014}"),
    ("measuredangle;", "\u{2221}"),
    ("mfr;", "\u{1D52A}"),
    ("mho;", "\u{2127}"),
    ("micro", "\u{B5}"),
    ("micro;", "\u{B5}"),
    ("mid;", "\u{2223}"),
    ("midast;", "\u{2A}"),
    ("midcir;", "\u{2AF0}"),
    ("middot", "\u{B7}"),
    ("middot;", "\u{B7}"),
    ("minus;", "\u{2212}"),
    ("minusb;", "\u{229F}"),
    ("minusd;", "\u{2238}"),
    ("minusdu;", "\u{2A2A}"),
    ("mlcp;", "\u{2ADB}"),
    ("mldr;", "\u{2026}"),
    ("mnplus;", "\u{2213}"),
    ("models;", "\u{22A7}"),
    ("mopf;", "\u{1D55E}"),
    ("mp;", "\u{2213}"),
    ("mscr;", "\u{1D4C2}"),
    ("mstpos;", "\u{223E}"),
    ("mu;", "\u{3BC}"),
    ("multimap;", "\u{22B8}"),
    ("mumap;", "\u{22B8}"),
    ("nGg;", "\u{22D9}\u{338}"),
    ("nGt;", "\u{226B}\u{20D2}"),
    ("nGtv;", "\u{226B}\u{338}"),
    ("nLeftarrow;", "\u{21CD}"),
    ("nLeftrightarrow;", "\u{21CE}"),
    ("nLl;", "\u{22D8}\u{338}"),
    ("nLt;", "\u{226A}\u{20D2}"),
    ("nLtv;", "\u{226A}\u{338}"),
    ("nRightarrow;", "\u{21CF}"),
    ("nVDash;", "\u{22AF}"),
    ("nVdash;", "\u{22AE}"),
    ("nabla;", "\u{2207}"),
    ("nacute;", "\u{144}"),
    ("nang;", "\u{2220}\u{20D2}"),
    ("nap;", "\u{2249}"),
    ("napE;", "\u{2A70}\u{338}"),
    ("napid;", "\u{224B}\u{338}"),
    ("napos;", "\u{149}"),
    ("napprox;", "\u{2249}"),
    ("natur;", "\u{266E}"),
    ("natural;", "\u{266E}"),
    ("naturals;", "\u{2115}"),
    ("nbsp", "\u{A0}"),
    ("nbsp;", "\u{A0}"),
    ("nbump;", "\u{224E}\u{338}"),
    ("nbumpe;", "\u{224F}\u{338}"),
    ("ncap;", "\u{2A43}"),
    ("ncaron;", "\u{148}"),
    ("ncedil;", "\u{146}"),
    ("ncong;", "\u{2247}"),
    ("ncongdot;", "\u{2A6D}\u{338}"),
    ("ncup;", "\u{2A42}"),
    ("ncy;", "\u{43D}"),
    ("ndash;", "\u{2013}"),
    ("ne;", "\u{2260}"),
    ("neArr;", "\u{21D7}"),
    ("nearhk;", "\u{2924}"),
    ("nearr;", "\u{2197}"),
    ("nearrow;", "\u{2197}"),
    ("nedot;", "\u{2250}\u{338}"),
    ("nequiv;", "\u{2262}"),
    ("nesear;", "\u{2928}"),
    ("nesim;", "\u{2242}\u{338}"),
    ("nexist;", "\u{2204}"),
    ("nexists;", "\u{2204}"),
    ("nfr;", "\u{1D52B}"),
    ("ngE;", "\u{2267}\u{338}"),
    ("nge;", "\u{2271}"),
    ("ngeq;", "\u{2271}"),
    ("ngeqq;", "\u{2267}\u{338}"),
    ("ngeqslant;", "\u{2A7E}\u{338}"),
    ("nges;", "\u{2A7E}\u{338}"),
    ("ngsim;", "\u{2275}"),
    ("ngt;", "\u{226F}"),
    ("ngtr;", "\u{226F}"),
    ("nhArr;", "\u{21CE}"),
    ("nharr;", "\u{21AE}"),
    ("nhpar;", "\u{2AF2}"),
    ("ni;", "\u{220B}"),
    ("nis;", "\u{22FC}"),
    ("nisd;", "\u{22FA}"),
    ("niv;", "\u{220B}"),
    ("njcy;", "\u{45A}"),
    ("nlArr;", "\u{21CD}"),
    ("nlE;", "\u{2266}\u{338}"),
    ("nlarr;", "\u{219A}"),
    ("nldr;", "\u{2025}"),
    ("nle;", "\u{2270}"),
    ("nleftarrow;", "\u{219A}"),
    ("nleftrightarrow;", "\u{21AE}"),
    ("nleq;", "\u{2270}"),
    ("nleqq;", "\u{2266}\u{338}"),
    ("nleqslant;", "\u{2A7D}\u{338}"),
    ("nles;", "\u{2A7D}\u{338}"),
    ("nless;", "\u{226E}"),
    ("nlsim;", "\u{2274}"),
    ("nlt;", "\u{226E}"),
    ("nltri;", "\u{22EA}"),
    ("nltrie;", "\u{22EC}"),
    ("nmid;", "\u{2224}"),
    ("nopf;", "\u{1D55F}"),
    ("not", "\u{AC}"),
    ("not;", "\u{AC}"),
    ("notin;", "\u{2209}"),
    ("notinE;", "\u{22F9}\u{338}"),
    ("notindot;", "\u{22F5}\u{338}"),
    ("notinva;", "\u{2209}"),
    ("notinvb;", "\u{22F7}"),
    ("notinvc;", "\u{22F6}"),
    ("notni;", "\u{220C}"),
    ("notniva;", "\u{220C}"),
    ("notnivb;", "\u{22FE}"),
    ("notnivc;", "\u{22FD}"),
    ("npar;", "\u{2226}"),
    ("nparallel;", "\u{2226}"),
    ("nparsl;", "\u{2AFD}\u{20E5}"),
    ("npart;", "\u{2202}\u{338}"),
    ("npolint;", "\u{2A14}"),
    ("npr;", "\u{2280}"),
    ("nprcue;", "\u{22E0}"),
    ("npre;", "\u{2AAF}\u{338}"),
    ("nprec;", "\u{2280}"),
    ("npreceq;", "\u{2AAF}\u{338}"),
    ("nrArr;", "\u{21CF}"),
    ("nrarr;", "\u{219B}"),
    ("nrarrc;", "\u{2933}\u{338}"),
    ("nrarrw;", "\u{219D}\u{338}"),
    ("nrightarrow;", "\u{219B}"),
    ("nrtri;", "\u{22EB}"),
    ("nrtrie;", "\u{22ED}"),
    ("nsc;", "\u{2281}"),
    ("nsccue;", "\u{22E1}"),
    ("nsce;", "\u{2AB0}\u{338}"),
    ("nscr;", "\u{1D4C3}"),
    ("nshortmid;", "\u{2224}"),
    ("nshortparallel;", "\u{2226}"),
    ("nsim;", "\u{2241}"),
    ("nsime;", "\u{2244}"),
    ("nsimeq;", "\u{2244}"),
    ("nsmid;", "\u{2224}"),
    ("nspar;", "\u{2226}"),
    ("nsqsube;", "\u{22E2}"),
    ("nsqsupe;", "\u{22E3}"),
    ("nsub;", "\u{2284}"),
    ("nsubE;", "\u{2AC5}\u{338}"),
    ("nsube;", "\u{2288}"),
    ("nsubset;", "\u{2282}\u{20D2}"),
    ("nsubseteq;", "\u{2288}"),
    ("nsubseteqq;", "\u{2AC5}\u{338}"),
    ("nsucc;", "\u{2281}"),
    ("nsucceq;", "\u{2AB0}\u{338}"),
    ("nsup;", "\u{2285}"),
    ("nsupE;", "\u{2AC6}\u{338}"),
    ("nsupe;", "\u{2289}"),
    ("nsupset;", "\u{2283}\u{20D2}"),
    ("nsupseteq;", "\u{2289}"),
    ("nsupseteqq;", "\u{2AC6}\u{338}"),
    ("ntgl;", "\u{2279}"),
    ("ntilde", "\u{F1}"),
    ("ntilde;", "\u{F1}"),
    ("ntlg;", "\u{2278}"),
    ("ntriangleleft;", "\u{22EA}"),
    ("ntrianglelefteq;", "\u{22EC}"),
    ("ntriangleright;", "\u{22EB}"),
    ("ntrianglerighteq;", "\u{22ED}"),
    ("nu;", "\u{3BD}"),
    ("num;", "\u{23}"),
    ("numero;", "\u{2116}"),
    ("numsp;", "\u{2007}"),
    ("nvDash;", "\u{22AD}"),
    ("nvHarr;", "\u{2904}"),
    ("nvap;", "\u{224D}\u{20D2}"),
    ("nvdash;", "\u{22AC}"),
    ("nvge;", "\u{2265}\u{20D2}"),
    ("nvgt;", "\u{3E}\u{20D2}"),
    ("nvinfin;", "\u{29DE}"),
    ("nvlArr;", "\u{2902}"),
    ("nvle;", "\u{2264}\u{20D2}"),
    ("nvlt;", "\u{3C}\u{20D2}"),
    ("nvltrie;", "\u{22B4}\u{20D2}"),
    ("nvrArr;", "\u{2903}"),
    ("nvrtrie;", "\u{22B5}\u{20D2}"),
    ("nvsim;", "\u{223C}\u{20D2}"),
    ("nwArr;", "\u{21D6}"),
    ("nwarhk;", "\u{2923}"),
    ("nwarr;", "\u{2196}"),
    ("nwarrow;", "\u{2196}"),
    ("nwnear;", "\u{2927}"),
    ("oS;", "\u{24C8}"),
    ("oacute", "\u{F3}"),
    ("oacute;", "\u{F3}"),
    ("oast;", "\u{229B}"),
    ("ocir;", "\u{229A}"),
    ("ocirc", "\u{F4}"),
    ("ocirc;", "\u{F4}"),
    ("ocy;", "\u{43E}"),
    ("odash;", "\u{229D}"),
    ("odblac;", "\u{151}"),
    ("odiv;", "\u{2A38}"),
    ("odot;", "\u{2299}"),
    ("odsold;", "\u{29BC}"),
    ("oelig;", "\u{153}"),
    ("ofcir;", "\u{29BF}"),
    ("ofr;", "\u{1D52C}"),
    ("ogon;", "\u{2DB}"),
    ("ograve", "\u{F2}"),
    ("ograve;", "\u{F2}"),
    ("ogt;", "\u{29C1}"),
    ("ohbar;", "\u{29B5}"),
    ("ohm;", "\u{3A9}"),
    ("oint;", "\u{222E}"),
    ("olarr;", "\u{21BA}"),
    ("olcir;", "\u{29BE}"),
    ("olcross;", "\u{29BB}"),
    ("oline;", "\u{203E}"),
    ("olt;", "\u{29C0}"),
    ("omacr;", "\u{14D}"),
    ("omega;", "\u{3C9}"),
    ("omicron;", "\u{3BF}"),
    ("omid;", "\u{29B6}"),
    ("ominus;", "\u{2296}"),
    ("oopf;", "\u{1D560}"),
    ("opar;", "\u{29B7}"),
    ("operp;", "\u{29B9}"),
    ("oplus;", "\u{2295}"),
    ("or;", "\u{2228}"),
    ("orarr;", "\u{21BB}"),
    ("ord;", "\u{2A5D}"),
    ("order;", "\u{2134}"),
    ("orderof;", "\u{2134}"),
    ("ordf", "\u{AA}"),
    ("ordf;", "\u{AA}"),
    ("ordm", "\u{BA}"),
    ("ordm;", "\u{BA}"),
    ("origof;", "\u{22B6}"),
    ("oror;", "\u{2A56}"),
    ("orslope;", "\u{2A57}"),
    ("orv;", "\u{2A5B}"),
    ("oscr;", "\u{2134}"),
    ("oslash", "\u{F8}"),
    ("oslash;", "\u{F8}"),
    ("osol;", "\u{2298}"),
    ("otilde", "\u{F5}"),
    ("otilde;", "\u{F5}"),
    ("otimes;", "\u{2297}"),
    ("otimesas;", "\u{2A36}"),
    ("ouml", "\u{F6}"),
    ("ouml;", "\u{F6}"),
    ("ovbar;", "\u{233D}"),
    ("par;", "\u{2225}"),
    ("para", "\u{B6}"),
    ("para;", "\u{B6}"),
    ("parallel;", "\u{2225}"),
    ("parsim;", "\u{2AF3}"),
    ("parsl;", "\u{2AFD}"),
    ("part;", "\u{2202}"),
    ("pcy;", "\u{43F}"),
    ("percnt;", "\u{25}"),
    ("period;", "\u{2E}"),
    ("permil;", "\u{2030}"),
    ("perp;", "\u{22A5}"),
    ("pertenk;", "\u{2031}"),
    ("pfr;", "\u{1D52D}"),
    ("phi;", "\u{3C6}"),
    ("phiv;", "\u{3D5}"),
    ("phmmat;", "\u{2133}"),
    ("phone;", "\u{260E}"),
    ("pi;", "\u{3C0}"),
    ("pitchfork;", "\u{22D4}"),
    ("piv;", "\u{3D6}"),
    ("planck;", "\u{210F}"),
    ("planckh;", "\u{210E}"),
    ("plankv;", "\u{210F}"),
    ("plus;", "\u{2B}"),
    ("plusacir;", "\u{2A23}"),
    ("plusb;", "\u{229E}"),
    ("pluscir;", "\u{2A22}"),
    ("plusdo;", "\u{2214}"),
    ("plusdu;", "\u{2A25}"),
    ("pluse;", "\u{2A72}"),
    ("plusmn", "\u{B1}"),
    ("plusmn;", "\u{B1}"),
    ("plussim;", "\u{2A26}"),
    ("plustwo;", "\u{2A27}"),
    ("pm;", "\u{B1}"),
    ("pointint;", "\u{2A15}"),
    ("popf;", "\u{1D561}"),
    ("pound", "\u{A3}"),
    ("pound;", "\u{A3}"),
    ("pr;", "\u{227A}"),
    ("prE;", "\u{2AB3}"),
    ("prap;", "\u{2AB7}"),
    ("prcue;", "\u{227C}"),
    ("pre;", "\u{2AAF}"),
    ("prec;", "\u{227A}"),
    ("precapprox;", "\u{2AB7}"),
    ("preccurlyeq;", "\u{227C}"),
    ("preceq;", "\u{2AAF}"),
    ("precnapprox;", "\u{2AB9}"),
    ("precneqq;", "\u{2AB5}"),
    ("precnsim;", "\u{22E8}"),
    ("precsim;", "\u{227E}"),
    ("prime;", "\u{2032}"),
    ("primes;", "\u{2119}"),
    ("prnE;", "\u{2AB5}"),
    ("prnap;", "\u{2AB9}"),
    ("prnsim;", "\u{22E8}"),
    ("prod;", "\u{220F}"),
    ("profalar;", "\u{232E}"),
    ("profline;", "\u{2312}"),
    ("profsurf;", "\u{2313}"),
    ("prop;", "\u{221D}"),
    ("propto;", "\u{221D}"),
    ("prsim;", "\u{227E}"),
    ("prurel;", "\u{22B0}"),
    ("pscr;", "\u{1D4C5}"),
    ("psi;", "\u{3C8}"),
    ("puncsp;", "\u{2008}"),
    ("qfr;", "\u{1D52E}"),
    ("qint;", "\u{2A0C}"),
    ("qopf;", "\u{1D562}"),
    ("qprime;", "\u{2057}"),
    ("qscr;", "\u{1D4C6}"),
    ("quaternions;", "\u{210D}"),
    ("quatint;", "\u{2A16}"),
    ("quest;", "\u{3F}"),
    ("questeq;", "\u{225F}"),
    ("quot", "\u{22}"),
    ("quot;", "\u{22}"),
    ("rAarr;", "\u{21DB}"),
    ("rArr;", "\u{21D2}"),
    ("rAtail;", "\u{291C}"),
    ("rBarr;", "\u{290F}"),
    ("rHar;", "\u{2964}"),
    ("race;", "\u{223D}\u{331}"),
    ("racute;", "\u{155}"),
    ("radic;", "\u{221A}"),
    ("raemptyv;", "\u{29B3}"),
    ("rang;", "\u{27E9}"),
    ("rangd;", "\u{2992}"),
    ("range;", "\u{29A5}"),
    ("rangle;", "\u{27E9}"),
    ("raquo", "\u{BB}"),
    ("raquo;", "\u{BB}"),
    ("rarr;", "\u{2192}"),
    ("rarrap;", "\u{2975}"),
    ("rarrb;", "\u{21E5}"),
    ("rarrbfs;", "\u{2920}"),
    ("rarrc;", "\u{2933}"),
    ("rarrfs;", "\u{291E}"),
    ("rarrhk;", "\u{21AA}"),
    ("rarrlp;", "\u{21AC}"),
    ("rarrpl;", "\u{2945}"),
    ("rarrsim;", "\u{2974}"),
    ("rarrtl;", "\u{21A3}"),
    ("rarrw;", "\u{219D}"),
    ("ratail;", "\u{291A}"),
    ("ratio;", "\u{2236}"),
    ("rationals;", "\u{211A}"),
    ("rbarr;", "\u{290D}"),
    ("rbbrk;", "\u{2773}"),
    ("rbrace;", "\u{7D}"),
    ("rbrack;", "\u{5D}"),
    ("rbrke;", "\u{298C}"),
    ("rbrksld;", "\u{298E}"),
    ("rbrkslu;", "\u{2990}"),
    ("rcaron;", "\u{159}"),
    ("rcedil;", "\u{157}"),
    ("rceil;", "\u{2309}"),
    ("rcub;", "\u{7D}"),
    ("rcy;", "\u{440}"),
    ("rdca;", "\u{2937}"),
    ("rdldhar;", "\u{2969}"),
    ("rdquo;", "\u{201D}"),
    ("rdquor;", "\u{201D}"),
    ("rdsh;", "\u{21B3}"),
    ("real;", "\u{211C}"),
    ("realine;", "\u{211B}"),
    ("realpart;", "\u{211C}"),
    ("reals;", "\u{211D}"),
    ("rect;", "\u{25AD}"),
    ("reg", "\u{AE}"),
    ("reg;", "\u{AE}"),
    ("rfisht;", "\u{297D}"),
    ("rfloor;", "\u{230B}"),
    ("rfr;", "\u{1D52F}"),
    ("rhard;", "\u{21C1}"),
    ("rharu;", "\u{21C0}"),
    ("rharul;", "\u{296C}"),
    ("rho;", "\u{3C1}"),
    ("rhov;", "\u{3F1}"),
    ("rightarrow;", "\u{2192}"),
    ("rightarrowtail;", "\u{21A3}"),
    ("rightharpoondown;", "\u{21C1}"),
    ("rightharpoonup;", "\u{21C0}"),
    ("rightleftarrows;", "\u{21C4}"),
    ("rightleftharpoons;", "\u{21CC}"),
    ("rightrightarrows;", "\u{21C9}"),
    ("rightsquigarrow;", "\u{219D}"),
    ("rightthreetimes;", "\u{22CC}"),
    ("ring;", "\u{2DA}"),
    ("risingdotseq;", "\u{2253}"),
    ("rlarr;", "\u{21C4}"),
    ("rlhar;", "\u{21CC}"),
    ("rlm;", "\u{200F}"),
    ("rmoust;", "\u{23B1}"),
    ("rmoustache;", "\u{23B1}"),
    ("rnmid;", "\u{2AEE}"),
    ("roang;", "\u{27ED}"),
    ("roarr;", "\u{21FE}"),
    ("robrk;", "\u{27E7}"),
    ("ropar;", "\u{2986}"),
    ("ropf;", "\u{1D563}"),
    ("roplus;", "\u{2A2E}"),
    ("rotimes;", "\u{2A35}"),
    ("rpar;", "\u{29}"),
    ("rpargt;", "\u{2994}"),
    ("rppolint;", "\u{2A12}"),
    ("rrarr;", "\u{21C9}"),
    ("rsaquo;", "\u{203A}"),
    ("rscr;", "\u{1D4C7}"),
    ("rsh;", "\u{21B1}"),
    ("rsqb;", "\u{5D}"),
    ("rsquo;", "\u{2019}"),
    ("rsquor;", "\u{2019}"),
    ("rthree;", "\u{22CC}"),
    ("rtimes;", "\u{22CA}"),
    ("rtri;", "\u{25B9}"),
    ("rtrie;", "\u{22B5}"),
    ("rtrif;", "\u{25B8}"),
    ("rtriltri;", "\u{29CE}"),
    ("ruluhar;", "\u{2968}"),
    ("rx;", "\u{211E}"),
    ("sacute;", "\u{15B}"),
    ("sbquo;", "\u{201A}"),
    ("sc;", "\u{227B}"),
    ("scE;", "\u{2AB4}"),
    ("scap;", "\u{2AB8}"),
    ("scaron;", "\u{161}"),
    ("sccue;", "\u{227D}"),
    ("sce;", "\u{2AB0}"),
    ("scedil;", "\u{15F}"),
    ("scirc;", "\u{15D}"),
    ("scnE;", "\u{2AB6}"),
    ("scnap;", "\u{2ABA}"),
    ("scnsim;", "\u{22E9}"),
    ("scpolint;", "\u{2A13}"),
    ("scsim;", "\u{227F}"),
    ("scy;", "\u{441}"),
    ("sdot;", "\u{22C5}"),
    ("sdotb;", "\u{22A1}"),
    ("sdote;", "\u{2A66}"),
    ("seArr;", "\u{21D8}"),
    ("searhk;", "\u{2925}"),
    ("searr;", "\u{2198}"),
    ("searrow;", "\u{2198}"),
    ("sect", "\u{A7}"),
    ("sect;", "\u{A7}"),
    ("semi;", "\u{3B}"),
    ("seswar;", "\u{2929}"),
    ("setminus;", "\u{2216}"),
    ("setmn;", "\u{2216}"),
    ("sext;", "\u{2736}"),
    ("sfr;", "\u{1D530}"),
    ("sfrown;", "\u{2322}"),
    ("sharp;", "\u{266F}"),
    ("shchcy;", "\u{449}"),
    ("shcy;", "\u{448}"),
    ("shortmid;", "\u{2223}"),
    ("shortparallel;", "\u{2225}"),
    ("shy", "\u{AD}"),
    ("shy;", "\u{AD}"),
    ("sigma;", "\u{3C3}"),
    ("sigmaf;", "\u{3C2}"),
    ("sigmav;", "\u{3C2}"),
    ("sim;", "\u{223C}"),
    ("simdot;", "\u{2A6A}"),
    ("sime;", "\u{2243}"),
    ("simeq;", "\u{2243}"),
    ("simg;", "\u{2A9E}"),
    ("simgE;", "\u{2AA0}"),
    ("siml;", "\u{2A9D}"),
    ("simlE;", "\u{2A9F}"),
    ("simne;", "\u{2246}"),
    ("simplus;", "\u{2A24}"),
    ("simrarr;", "\u{2972}"),
    ("slarr;", "\u{2190}"),
    ("smallsetminus;", "\u{2216}"),
    ("smashp;", "\u{2A33}"),
    ("smeparsl;", "\u{29E4}"),
    ("smid;", "\u{2223}"),
    ("smile;", "\u{2323}"),
    ("smt;", "\u{2AAA}"),
    ("smte;", "\u{2AAC}"),
    ("smtes;", "\u{2AAC}\u{FE00}"),
    ("softcy;", "\u{44C}"),
    ("sol;", "\u{2F}"),
    ("solb;", "\u{29C4}"),
    ("solbar;", "\u{233F}"),
    ("sopf;", "\u{1D564}"),
    ("spades;", "\u{2660}"),
    ("spadesuit;", "\u{2660}"),
    ("spar;", "\u{2225}"),
    ("sqcap;", "\u{2293}"),
    ("sqcaps;", "\u{2293}\u{FE00}"),
    ("sqcup;", "\u{2294}"),
    ("sqcups;", "\u{2294}\u{FE00}"),
    ("sqsub;", "\u{228F}"),
    ("sqsube;", "\u{2291}"),
    ("sqsubset;", "\u{228F}"),
    ("sqsubseteq;", "\u{2291}"),
    ("sqsup;", "\u{2290}"),
    ("sqsupe;", "\u{2292}"),
    ("sqsupset;", "\u{2290}"),
    ("sqsupseteq;", "\u{2292}"),
    ("squ;", "\u{25A1}"),
    ("square;", "\u{25A1}"),
    ("squarf;", "\u{25AA}"),
    ("squf;", "\u{25AA}"),
    ("srarr;", "\u{2192}"),
    ("sscr;", "\u{1D4C8}"),
    ("ssetmn;", "\u{2216}"),
    ("ssmile;", "\u{2323}"),
    ("sstarf;", "\u{22C6}"),
    ("star;", "\u{2606}"),
    ("starf;", "\u{2605}"),
    ("straightepsilon;", "\u{3F5}"),
    ("straightphi;", "\u{3D5}"),
    ("strns;", "\u{AF}"),
    ("sub;", "\u{2282}"),
    ("subE;", "\u{2AC5}"),
    ("subdot;", "\u{2ABD}"),
    ("sube;", "\u{2286}"),
    ("subedot;", "\u{2AC3}"),
    ("submult;", "\u{2AC1}"),
    ("subnE;", "\u{2ACB}"),
    ("subne;", "\u{228A}"),
    ("subplus;", "\u{2ABF}"),
    ("subrarr;", "\u{2979}"),
    ("subset;", "\u{2282}"),
    ("subseteq;", "\u{2286}"),
    ("subseteqq;", "\u{2AC5}"),
    ("subsetneq;", "\u{228A}"),
    ("subsetneqq;", "\u{2ACB}"),
    ("subsim;", "\u{2AC7}"),
    ("subsub;", "\u{2AD5}"),
    ("subsup;", "\u{2AD3}"),
    ("succ;", "\u{227B}"),
    ("succapprox;", "\u{2AB8}"),
    ("succcurlyeq;", "\u{227D}"),
    ("succeq;", "\u{2AB0}"),
    ("succnapprox;", "\u{2ABA}"),
    ("succneqq;", "\u{2AB6}"),
    ("succnsim;", "\u{22E9}"),
    ("succsim;", "\u{227F}"),
    ("sum;", "\u{2211}"),
    ("sung;", "\u{266A}"),
    ("sup1", "\u{B9}"),
    ("sup1;", "\u{B9}"),
    ("sup2", "\u{B2}"),
    ("sup2;", "\u{B2}"),
    ("sup3", "\u{B3}"),
    ("sup3;", "\u{B3}"),
    ("sup;", "\u{2283}"),
    ("supE;", "\u{2AC6}"),
    ("supdot;", "\u{2ABE}"),
    ("supdsub;", "\u{2AD8}"),
    ("supe;", "\u{2287}"),
    ("supedot;", "\u{2AC4}"),
    ("suphsol;", "\u{27C9}"),
    ("suphsub;", "\u{2AD7}"),
    ("suplarr;", "\u{297B}"),
    ("supmult;", "\u{2AC2}"),
    ("supnE;", "\u{2ACC}"),
    ("supne;", "\u{228B}"),
    ("supplus;", "\u{2AC0}"),
    ("supset;", "\u{2283}"),
    ("supseteq;", "\u{2287}"),
    ("supseteqq;", "\u{2AC6}"),
    ("supsetneq;", "\u{228B}"),
    ("supsetneqq;", "\u{2ACC}"),
    ("supsim;", "\u{2AC8}"),
    ("supsub;", "\u{2AD4}"),
    ("supsup;", "\u{2AD6}"),
    ("swArr;", "\u{21D9}"),
    ("swarhk;", "\u{2926}"),
    ("swarr;", "\u{2199}"),
    ("swarrow;", "\u{2199}"),
    ("swnwar;", "\u{292A}"),
    ("szlig", "\u{DF}"),
    ("szlig;", "\u{DF}"),
    ("target;", "\u{2316}"),
    ("tau;", "\u{3C4}"),
    ("tbrk;", "\u{23B4}"),
    ("tcaron;", "\u{165}"),
    ("tcedil;", "\u{163}"),
    ("tcy;", "\u{442}"),
    ("tdot;", "\u{20DB}"),
    ("telrec;", "\u{2315}"),
    ("tfr;", "\u{1D531}"),
    ("there4;", "\u{2234}"),
    ("therefore;", "\u{2234}"),
    ("theta;", "\u{3B8}"),
    ("thetasym;", "\u{3D1}"),
    ("thetav;", "\u{3D1}"),
    ("thickapprox;", "\u{2248}"),
    ("thicksim;", "\u{223C}"),
    ("thinsp;", "\u{2009}"),
    ("thkap;", "\u{2248}"),
    ("thksim;", "\u{223C}"),
    ("thorn", "\u{FE}"),
    ("thorn;", "\u{FE}"),
    ("tilde;", "\u{2DC}"),
    ("times", "\u{D7}"),
    ("times;", "\u{D7}"),
    ("timesb;", "\u{22A0}"),
    ("timesbar;", "\u{2A31}"),
    ("timesd;", "\u{2A30}"),
    ("tint;", "\u{222D}"),
    ("toea;", "\u{2928}"),
    ("top;", "\u{22A4}"),
    ("topbot;", "\u{2336}"),
    ("topcir;", "\u{2AF1}"),
    ("topf;", "\u{1D565}"),
    ("topfork;", "\u{2ADA}"),
    ("tosa;", "\u{2929}"),
    ("tprime;", "\u{2034}"),
    ("trade;", "\u{2122}"),
    ("triangle;", "\u{25B5}"),
    ("triangledown;", "\u{25BF}"),
    ("triangleleft;", "\u{25C3}"),
    ("trianglelefteq;", "\u{22B4}"),
    ("triangleq;", "\u{225C}"),
    ("triangleright;", "\u{25B9}"),
    ("trianglerighteq;", "\u{22B5}"),
    ("tridot;", "\u{25EC}"),
    ("trie;", "\u{225C}"),
    ("triminus;", "\u{2A3A}"),
    ("triplus;", "\u{2A39}"),
    ("trisb;", "\u{29CD}"),
    ("tritime;", "\u{2A3B}"),
    ("trpezium;", "\u{23E2}"),
    ("tscr;", "\u{1D4C9}"),
    ("tscy;", "\u{446}"),
    ("tshcy;", "\u{45B}"),
    ("tstrok;", "\u{167}"),
    ("twixt;", "\u{226C}"),
    ("twoheadleftarrow;", "\u{219E}"),
    ("twoheadrightarrow;", "\u{21A0}"),
    ("uArr;", "\u{21D1}"),
    ("uHar;", "\u{2963}"),
    ("uacute", "\u{FA}"),
    ("uacute;", "\u{FA}"),
    ("uarr;", "\u{2191}"),
    ("ubrcy;", "\u{45E}"),
    ("ubreve;", "\u{16D}"),
    ("ucirc", "\u{FB}"),
    ("ucirc;", "\u{FB}"),
    ("ucy;", "\u{443}"),
    ("udarr;", "\u{21C5}"),
    ("udblac;", "\u{171}"),
    ("udhar;", "\u{296E}"),
    ("ufisht;", "\u{297E}"),
    ("ufr;", "\u{1D532}"),
    ("ugrave", "\u{F9}"),
    ("ugrave;", "\u{F9}"),
    ("uharl;", "\u{21BF}"),
    ("uharr;", "\u{21BE}"),
    ("uhblk;", "\u{2580}"),
    ("ulcorn;", "\u{231C}"),
    ("ulcorner;", "\u{231C}"),
    ("ulcrop;", "\u{230F}"),
    ("ultri;", "\u{25F8}"),
    ("umacr;", "\u{16B}"),
    ("uml", "\u{A8}"),
    ("uml;", "\u{A8}"),
    ("uogon;", "\u{173}"),
    ("uopf;", "\u{1D566}"),
    ("uparrow;", "\u{2191}"),
    ("updownarrow;", "\u{2195}"),
    ("upharpoonleft;", "\u{21BF}"),
    ("upharpoonright;", "\u{21BE}"),
    ("uplus;", "\u{228E}"),
    ("upsi;", "\u{3C5}"),
    ("upsih;", "\u{3D2}"),
    ("upsilon;", "\u{3C5}"),
    ("upuparrows;", "\u{21C8}"),
    ("urcorn;", "\u{231D}"),
    ("urcorner;", "\u{231D}"),
    ("urcrop;", "\u{230E}"),
    ("uring;", "\u{16F}"),
    ("urtri;", "\u{25F9}"),
    ("uscr;", "\u{1D4CA}"),
    ("utdot;", "\u{22F0}"),
    ("utilde;", "\u{169}"),
    ("utri;", "\u{25B5}"),
    ("utrif;", "\u{25B4}"),
    ("uuarr;", "\u{21C8}"),
    ("uuml", "\u{FC}"),
    ("uuml;", "\u{FC}"),
    ("uwangle;", "\u{29A7}"),
    ("vArr;", "\u{21D5}"),
    ("vBar;", "\u{2AE8}"),
    ("vBarv;", "\u{2AE9}"),
    ("vDash;", "\u{22A8}"),
    ("vangrt;", "\u{299C}"),
    ("varepsilon;", "\u{3F5}"),
    ("varkappa;", "\u{3F0}"),
    ("varnothing;", "\u{2205}"),
    ("varphi;", "\u{3D5}"),
    ("varpi;", "\u{3D6}"),
    ("varpropto;", "\u{221D}"),
    ("varr;", "\u{2195}"),
    ("varrho;", "\u{3F1}"),
    ("varsigma;", "\u{3C2}"),
    ("varsubsetneq;", "\u{228A}\u{FE00}"),
    ("varsubsetneqq;", "\u{2ACB}\u{FE00}"),
    ("varsupsetneq;", "\u{228B}\u{FE00}"),
    ("varsupsetneqq;", "\u{2ACC}\u{FE00}"),
    ("vartheta;", "\u{3D1}"),
    ("vartriangleleft;", "\u{22B2}"),
    ("vartriangleright;", "\u{22B3}"),
    ("vcy;", "\u{432}"),
    ("vdash;", "\u{22A2}"),
    ("vee;", "\u{2228}"),
    ("veebar;", "\u{22BB}"),
    ("veeeq;", "\u{225A}"),
    ("vellip;", "\u{22EE}"),
    ("verbar;", "\u{7C}"),
    ("vert;", "\u{7C}"),
    ("vfr;", "\u{1D533}"),
    ("vltri;", "\u{22B2}"),
    ("vnsub;", "\u{2282}\u{20D2}"),
    ("vnsup;", "\u{2283}\u{20D2}"),
    ("vopf;", "\u{1D567}"),
    ("vprop;", "\u{221D}"),
    ("vrtri;", "\u{22B3}"),
    ("vscr;", "\u{1D4CB}"),
    ("vsubnE;", "\u{2ACB}\u{FE00}"),
    ("vsubne;", "\u{228A}\u{FE00}"),
    ("vsupnE;", "\u{2ACC}\u{FE00}"),
    ("vsupne;", "\u{228B}\u{FE00}"),
    ("vzigzag;", "\u{299A}"),
    ("wcirc;", "\u{175}"),
    ("wedbar;", "\u{2A5F}"),
    ("wedge;", "\u{2227}"),
    ("wedgeq;", "\u{2259}"),
    ("weierp;", "\u{2118}"),
    ("wfr;", "\u{1D534}"),
    ("wopf;", "\u{1D568}"),
    ("wp;", "\u{2118}"),
    ("wr;", "\u{2240}"),
    ("wreath;", "\u{2240}"),
    ("wscr;", "\u{1D4CC}"),
    ("xcap;", "\u{22C2}"),
    ("xcirc;", "\u{25EF}"),
    ("xcup;", "\u{22C3}"),
    ("xdtri;", "\u{25BD}"),
    ("xfr;", "\u{1D535}"),
    ("xhArr;", "\u{27FA}"),
    ("xharr;", "\u{27F7}"),
    ("xi;", "\u{3BE}"),
    ("xlArr;", "\u{27F8}"),
    ("xlarr;", "\u{27F5}"),
    ("xmap;", "\u{27FC}"),
    ("xnis;", "\u{22FB}"),
    ("xodot;", "\u{2A00}"),
    ("xopf;", "\u{1D569}"),
    ("xoplus;", "\u{2A01}"),
    ("xotime;", "\u{2A02}"),
    ("xrArr;", "\u{27F9}"),
    ("xrarr;", "\u{27F6}"),
    ("xscr;", "\u{1D4CD}"),
    ("xsqcup;", "\u{2A06}"),
    ("xuplus;", "\u{2A04}"),
    ("xutri;", "\u{25B3}"),
    ("xvee;", "\u{22C1}"),
    ("xwedge;", "\u{22C0}"),
    ("yacute", "\u{FD}"),
    ("yacute;", "\u{FD}"),
    ("yacy;", "\u{44F}"),
    ("ycirc;", "\u{177}"),
    ("ycy;", "\u{44B}"),
    ("yen", "\u{A5}"),
    ("yen;", "\u{A5}"),
    ("yfr;", "\u{1D536}"),
    ("yicy;", "\u{457}"),
    ("yopf;", "\u{1D56A}"),
    ("yscr;", "\u{1D4CE}"),
    ("yucy;", "\u{44E}"),
    ("yuml", "\u{FF}"),
    ("yuml;", "\u{FF}"),
    ("zacute;", "\u{17A}"),
    ("zcaron;", "\u{17E}"),
    ("zcy;", "\u{437}"),
    ("zdot;", "\u{17C}"),
    ("zeetrf;", "\u{2128}"),
    ("zeta;", "\u{3B6}"),
    ("zfr;", "\u{1D537}"),
    ("zhcy;", "\u{436}"),
    ("zigrarr;", "\u{21DD}"),
    ("zopf;", "\u{1D56B}"),
    ("zscr;", "\u{1D4CF}"),
    ("zwj;", "\u{200D}"),
    ("zwnj;", "\u{200C}"),
];

pub static HTML5_ENTITIES_REF: LazyLock<HashMap<&'static str, &'static str>> =
    LazyLock::new(|| HTML5_ENTITIES.iter().copied().collect());
