//
// Collection.swift
// mas
//
// Copyright © 2025 mas-cli. All rights reserved.
//

extension Collection {
	func dropLast(while predicate: (Element) throws -> Bool) rethrows -> SubSequence {
		try indices.reversed().first { try !predicate(self[$0]) }.map { self[...$0] } ?? self[endIndex...]
	}
}

extension Collection where Element: Sendable {
	func concurrentMap<T: Sendable>(
		maxConcurrentTaskCount: Int = defaultMaxConcurrentTaskCount,
		_ transform: @escaping @Sendable (Element) async -> T,
	) async -> [T] { // swiftlint:disable:next force_unwrapping
		await concurrentTransform(maxConcurrentTaskCount: maxConcurrentTaskCount, transform).map { $0! }
	}

	func concurrentMap<T: Sendable>(
		maxConcurrentTaskCount: Int = defaultMaxConcurrentTaskCount,
		_ transform: @escaping @Sendable (Element) async throws -> T,
	) async rethrows -> [T] { // periphery:ignore
		try await concurrentTransform(maxConcurrentTaskCount: maxConcurrentTaskCount, transform).map { $0! }
	} // swiftlint:disable:previous force_unwrapping

	func concurrentCompactMap<T: Sendable>(
		maxConcurrentTaskCount: Int = defaultMaxConcurrentTaskCount,
		_ transform: @escaping @Sendable (Element) async -> T?,
	) async -> [T] {
		await concurrentTransform(maxConcurrentTaskCount: maxConcurrentTaskCount, transform).compactMap(\.self)
	}

	func concurrentCompactMap<T: Sendable>(
		maxConcurrentTaskCount: Int = defaultMaxConcurrentTaskCount,
		_ transform: @escaping @Sendable (Element) async throws -> T?,
	) async rethrows -> [T] { // periphery:ignore
		try await concurrentTransform(maxConcurrentTaskCount: maxConcurrentTaskCount, transform).compactMap(\.self)
	}

	func concurrentCompactMap<T: Sendable, E: Error>(
		attemptingTo perform: String,
		maxConcurrentTaskCount: Int = defaultMaxConcurrentTaskCount,
		_ transform: @escaping @Sendable (Element) async throws(E) -> T?,
	) async -> [T] {
		await concurrentCompactMap(maxConcurrentTaskCount: maxConcurrentTaskCount) { element in
			do {
				return try await transform(element)
			} catch {
				MAS.printer.error(error is MASError ? [] : ["Failed to", perform, element], error: error)
				return nil
			}
		}
	}

	private func concurrentTransform<T: Sendable>(
		maxConcurrentTaskCount: Int,
		_ transform: @escaping @Sendable (Element) async throws -> T?,
	) async rethrows -> [T?] {
		try await withThrowingTaskGroup(of: (index: Int, result: T?).self) { group in
			var iterator = enumerated().makeIterator()
			func addNextTask() {
				if let next = iterator.next() {
					group.addTask {
						(next.offset, try await transform(next.element))
					}
				}
			}

			for _ in 0..<Swift.min(count, maxConcurrentTaskCount) {
				addNextTask()
			}

			return try await group.reduce(into: [T?](repeating: nil, count: count)) { results, indexedResult in
				results[indexedResult.index] = indexedResult.result
				addNextTask()
			}
		}
	}
}

private let defaultMaxConcurrentTaskCount = 16
