//! Qore namespace
namespace Qore {
//! Xml namespace
namespace Xml {
//! The SaxIterator class provides a SAX iterator for XML data based on <a href="http://xmlsoft.org">libxml2</a>
/***/
class SaxIterator : public AbstractIterator {

public:
//! creates a new SaxIterator object from the XML string and element name passed
/** @param xml an XML string to iterate
    @param element_name the name of the element to iterate through
    @param opts the following options are supported:
    - \c xml_parse_options: (int bitfield) XML parsing flags; see @ref xml_parsing_constants for more information
    - \c xml_input_io: (AbstractXmlIoInputCallback) an AbstractXmlIoInputCallback object to resolve external XSD schema references
    - \c xsd: (string) an XSD string for schema validation while parsing

    @par Example:
    @code
SaxIterator i(xml, "DetailRecord");
map printf("record %d: %y\n", $#, $1), i;
    @endcode

    @throw XMLDOC-CONSTRUCTOR-ERROR error parsing XML string

    @since xml 1.4 added support for the \a opts argument
*/
 constructor(string xml, string element_name, __7_ hash opts);

public:
//! Returns a copy of the current object (the copy will be reset to the beginning of the XML string)
/** @return a copy of the current object (the copy will be reset to the beginning of the XML string)

    @par Example:
    @code SaxIterator icopy = i.copy(); @endcode
*/
 copy();

public:
//! returns the current value or throws an \c INVALID-ITERATOR exception if the iterator is invalid
/** @return the current value or throws an \c INVALID-ITERATOR exception if the iterator is invalid

   @par Code Flags:
    @ref RET_VALUE_ONLY

     @par Example:
    @code
while (i.next()) {
    printf("+ %y\n", i.getValue());
}
    @endcode

    @throw INVALID-ITERATOR the iterator is not pointing at a valid element
    @throw ITERATOR-THREAD-ERROR this exception is thrown if this method is called from any thread other than the thread that created the object
 */
auto getValue();

public:
//! Moves the current position to the next element in the XML string; returns @ref False if there are no more elements; if the iterator is not pointing at a valid element before this call, the iterator will be positioned on the first element in the XML if the XML string contains the required element
/** This method will return @ref True again after it returns @ref False once if the XML string contains the target element, otherwise it will always return @ref False.
    The iterator object should not be used after this method returns @ref False

    @return @ref False if there are no more elements in the XML string (in which case the iterator object is invalid and should not be used); @ref True if successful (meaning that the iterator object is valid)

    @par Example:
    @code
while (i.next()) {
    printf(" + %y\n", i.getValue());
}
    @endcode

    @throw ITERATOR-THREAD-ERROR this exception is thrown if this method is called from any thread other than the thread that created the object
 */
bool next();

public:
//! Reset the iterator instance to its initial state
/** Reset the iterator instance to its initial state

   @par Example
   @code
i.reset();
   @endcode

    @throw ITERATOR-THREAD-ERROR this exception is thrown if this method is called from any thread other than the thread that created the object
 */
 reset();

public:
//! returns @ref Qore::True "True" if the iterator is currently pointing at a valid element, @ref Qore::False "False" if not
/** @return @ref Qore::True "True" if the iterator is currently pointing at a valid element, @ref Qore::False "False" if not

   @par Code Flags:
    @ref CONSTANT

     @par Example:
    @code
if (i.valid())
    printf("current value: %y\n", i.getValue());
    @endcode
 */
bool valid();
};
}
}
