use std::{env, process};
use wasi_tests::{assert_errno, create_tmp_dir, open_scratch_directory};

unsafe fn test_nofollow_errors(dir_fd: wasi::Fd) {
    // Create a directory for the symlink to point to.
    wasi::path_create_directory(dir_fd, "target").expect("creating a dir");

    // Create a symlink.
    wasi::path_symlink("target", dir_fd, "symlink").expect("creating a symlink");

    // Try to open it as a directory with O_NOFOLLOW again.
    assert_errno!(
        wasi::path_open(dir_fd, 0, "symlink", wasi::OFLAGS_DIRECTORY, 0, 0, 0)
            .expect_err("opening a directory symlink as a directory should fail"),
        wasi::ERRNO_LOOP,
        wasi::ERRNO_NOTDIR
    );

    // Try to open it with just O_NOFOLLOW.
    assert_errno!(
        wasi::path_open(dir_fd, 0, "symlink", 0, 0, 0, 0)
            .expect_err("opening a symlink with O_NOFOLLOW should fail"),
        wasi::ERRNO_LOOP,
        wasi::ERRNO_ACCES
    );

    // Try to open it as a directory without O_NOFOLLOW.
    let file_fd = wasi::path_open(
        dir_fd,
        wasi::LOOKUPFLAGS_SYMLINK_FOLLOW,
        "symlink",
        wasi::OFLAGS_DIRECTORY,
        0,
        0,
        0,
    )
    .expect("opening a symlink as a directory");
    assert!(
        file_fd > libc::STDERR_FILENO as wasi::Fd,
        "file descriptor range check",
    );
    wasi::fd_close(file_fd).expect("closing a file");

    // Replace the target directory with a file.
    wasi::path_unlink_file(dir_fd, "symlink").expect("removing a file");
    wasi::path_remove_directory(dir_fd, "target")
        .expect("remove_directory on a directory should succeed");

    let file_fd =
        wasi::path_open(dir_fd, 0, "target", wasi::OFLAGS_CREAT, 0, 0, 0).expect("creating a file");
    wasi::fd_close(file_fd).expect("closing a file");
    wasi::path_symlink("target", dir_fd, "symlink").expect("creating a symlink");

    // Try to open it as a directory with O_NOFOLLOW again.
    assert_errno!(
        wasi::path_open(dir_fd, 0, "symlink", wasi::OFLAGS_DIRECTORY, 0, 0, 0)
            .expect_err("opening a directory symlink as a directory should fail"),
        wasi::ERRNO_LOOP,
        wasi::ERRNO_NOTDIR
    );

    // Try to open it with just O_NOFOLLOW.
    assert_errno!(
        wasi::path_open(dir_fd, 0, "symlink", 0, 0, 0, 0)
            .expect_err("opening a symlink with NOFOLLOW should fail"),
        wasi::ERRNO_LOOP
    );

    // Try to open it as a directory without O_NOFOLLOW.
    assert_errno!(
        wasi::path_open(
            dir_fd,
            wasi::LOOKUPFLAGS_SYMLINK_FOLLOW,
            "symlink",
            wasi::OFLAGS_DIRECTORY,
            0,
            0,
            0,
        )
        .expect_err("opening a symlink to a file as a directory"),
        wasi::ERRNO_NOTDIR
    );

    // Clean up.
    wasi::path_unlink_file(dir_fd, "target").expect("removing a file");
    wasi::path_unlink_file(dir_fd, "symlink").expect("removing a file");
}

fn main() {
    let mut args = env::args();
    let prog = args.next().unwrap();
    let arg = if let Some(arg) = args.next() {
        arg
    } else {
        eprintln!("usage: {} <scratch directory>", prog);
        process::exit(1);
    };

    // Open scratch directory
    let base_dir_fd = match open_scratch_directory(&arg) {
        Ok(dir_fd) => dir_fd,
        Err(err) => {
            eprintln!("{}", err);
            process::exit(1)
        }
    };

    const DIR_NAME: &str = "nofollow_errors_dir.cleanup";
    let dir_fd;
    unsafe {
        dir_fd = create_tmp_dir(base_dir_fd, DIR_NAME);
    }

    // Run the tests.
    unsafe { test_nofollow_errors(dir_fd) }

    unsafe { wasi::path_remove_directory(base_dir_fd, DIR_NAME).expect("failed to remove dir") }
}
