#!/usr/bin/env python3
#
# Generate the Zeek file containing the current list of known
# Certificate Transparency logs from the source file provided
# by Google.
#

import base64
import hashlib
import json
import sys
import urllib.request
from datetime import datetime

# This is the only kind-of user-configurable line
google_log_url = "https://www.gstatic.com/ct/log_list/v3/log_list.json"

try:
    with urllib.request.urlopen(google_log_url) as r:
        google_known_logs_json = r.read()
except urllib.error.URLError:
    sys.exit(f"Could not get {google_log_url}")

try:
    logs_json = json.loads(google_known_logs_json)
except:
    sys.exit(f"Could not decide json from request for {google_log_url}")

print("#")
print("# Do not edit this file. This file is automatically generated by gen-ct-list.py")
print(f"# File generated at {datetime.now().strftime('%a %b %d %H:%M:%S %Y')}")
print(f"# File generated from {google_log_url}")
print(f"# Source file generated at: {logs_json.get('log_list_timestamp')}")
print(f"# Source file version: {logs_json.get('version')}")
print("#")
print("")
print("@load base/protocols/ssl")
print("module SSL;")
print("")
print("## @docs-omit-value")
print("redef ct_logs += {")


def unpack_to_escaped_hex(data):
    key_hexes = [a + b for a, b in zip(data[::2], data[1::2])]
    return "\\x".join(["", *key_hexes])


for operator in logs_json.get("operators", []):
    opname = operator.get("name")
    for log in operator.get("logs", []):
        key = unpack_to_escaped_hex(base64.b64decode(log.get("key")).hex())
        logid = unpack_to_escaped_hex(
            hashlib.sha256(base64.b64decode(log.get("key"))).hexdigest()
        )
        mmd = log.get("mmd")
        url = log.get("url")
        desc = log.get("description")
        print(
            f'["{logid}"] = CTInfo($description="{desc}", $operator="{opname}", $url="{url}", $maximum_merge_delay={mmd}, $key="{key}"),'
        )

print("};")
